#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
#

import sys

from checkmarx.task import *
from checkmarx.task.helper.risk_profile import *

try:
    validate_mandatory_field(checkmarxServer, "Checkmarx server ID must be provided")
    validate_mandatory_field(team, "Missing value for Team")
    validate_mandatory_field(projectName, "Missing value for Project Name")

    checkmarx_team_id = CheckmarxService.get_team_id_by_name(checkmarx_server=checkmarxServer, team_name=team)

    project_list = CheckmarxService.get_project(checkmarx_server=checkmarxServer, project_name=projectName,
                                                team_id=checkmarx_team_id)
    if project_list:
        checkmarx_project_id = project_list[0]["id"]
    else:
        raise Exception("Could not get project data for : %s" % projectName)

    if not scanId:
        osa_scan_status = CheckmarxService.get_latest_osa_scan(checkmarx_server=checkmarxServer,
                                                                   project_id=checkmarx_project_id)
        internal_scan_id = osa_scan_status["id"]
    else:
        internal_scan_id = scanId
        osa_scan_status = CheckmarxService.get_osa_scan_status(checkmarx_server=checkmarxServer,
                                                               scan_id=internal_scan_id)
        if osa_scan_status["state"]["name"] != BaseStatus.BASE_STATUS_SUCCEEDED:
            add_comment("CxOSA Scan is in progress or failed for scan id : %s. Can not retrieve result." % internal_scan_id)
            sys.exit(1)

    osa_scan_result = CheckmarxService.get_osa_results_stats(checkmarx_server=checkmarxServer,
                                                             scan_id=internal_scan_id)
    osa_scan_library_data = CheckmarxService.get_osa_libraries_data(checkmarx_server=checkmarxServer,
                                                                    scan_id=internal_scan_id)

    risk_high = Risk(osa_scan_result["highVulnerabilityLibraries"], securityRiskHighThreshold, "High")
    risk_medium = Risk(osa_scan_result["mediumVulnerabilityLibraries"], securityRiskMediumThreshold, "Medium")
    risk_low = Risk(osa_scan_result["lowVulnerabilityLibraries"], securityRiskLowThreshold, "Low")

    license_risk_high = Risk(osa_scan_library_data["highRisk"], licenseRiskHighThreshold, "High")
    license_risk_medium = Risk(osa_scan_library_data["mediumRisk"], licenseRiskMediumThreshold, "Medium")
    license_risk_unknown = Risk(osa_scan_library_data["unknownRisk"], licenseRiskUnknownThreshold, "Unknown")

    security_risk_profile = RiskProfile(risk_high, risk_medium, risk_low)
    license_risk_profile = RiskProfile(license_risk_high, license_risk_medium, license_risk_unknown)
    print("Scan Id : " + internal_scan_id + "\n")
    print("**Security Risk**\n&nbsp;\n")
    status = "| Severity       |      Value       |  Threshold    |   Status  | \n"
    status += "|--------------- |:----------------:|:-------------:|:---------:| \n"
    status += risk_high.get_display_text() + "\n"
    status += risk_medium.get_display_text() + "\n"
    status += risk_low.get_display_text() + "\n"
    status += "\n&nbsp;\n"
    print(status)
    print("**License Risk**\n&nbsp;\n")
    status = "| Severity       |      Value       |  Threshold    |   Status  | \n"
    status += "|--------------- |:----------------:|:-------------:|:---------:| \n"
    status += license_risk_high.get_display_text() + "\n"
    status += license_risk_medium.get_display_text() + "\n"
    status += license_risk_unknown.get_display_text() + "\n"
    status += "\n&nbsp;\n"
    print(status)

    compliance_data = {}

    for security_risk in [risk_high, risk_medium, risk_low]:
        compliance_data["Security " + str(security_risk.get_type())] = {
            "Value": str(security_risk.get_value()),
            "Threshold": str(security_risk.get_threshold()),
            "Status": "Pass" if security_risk.validate() else "Fail"
        }

    for license_risk in [license_risk_high, license_risk_medium, license_risk_unknown]:
        compliance_data["License " + str(license_risk.get_type())] = {
            "Value": str(license_risk.get_value()),
            "Threshold": str(license_risk.get_threshold()),
            "Status": "Pass" if license_risk.validate() else "Fail"
        }
    compliance_data["ScanId"] = str(internal_scan_id)
    analysis_date = osa_scan_status["startAnalyzeTime"]

    if security_risk_profile.validate() and license_risk_profile.validate():
        add_code_compliance_record(taskReportingApi, task, checkmarxServer, project=projectName, project_url="Not available",
                                  outcome="PASSED", compliance_data=compliance_data, analysis_date=analysis_date)
    else:
        add_comment("Issues count is higher than allowed threshold")
        add_code_compliance_record(taskReportingApi, task, checkmarxServer, project=projectName, project_url="Not available",
                                  outcome="FAILED", compliance_data=compliance_data, analysis_date=analysis_date)
        sys.exit(1)

except Exception as e:
    add_comment("Exception occurred: %s " % e)
    sys.exit(1)
