#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
#

import sys

from checkmarx.task import *
from checkmarx.task.helper.risk_profile import *


try:
    validate_mandatory_field(checkmarxScaServer, "Checkmarx SCA server ID must be provided")
    validate_mandatory_field(projectName, "Missing value for Project Name")

    if checkmarxScaServer['serverEnvironment'] == "US":
        checkmarxScaServer['url'] = "https://api-sca.checkmarx.net"
    else:
        checkmarxScaServer['url'] = "https://eu.api-sca.checkmarx.net"

    project_details = CheckmarxService.get_sca_project(checkmarx_server=checkmarxScaServer, project_name=projectName)
    if project_details:
        checkmarx_project_id = project_details["id"]
    else:
        raise Exception("Could not get project data for : %s" % projectName)

    risk_reports = CheckmarxService.get_sca_risk_reports(checkmarx_server=checkmarxScaServer, project_id=checkmarx_project_id)
    risk_report = risk_reports[0]

    risk_high = Risk(risk_report["highVulnerabilityCount"], securityRiskHighThreshold, "High")
    risk_medium = Risk(risk_report["mediumVulnerabilityCount"], securityRiskMediumThreshold, "Medium")
    risk_low = Risk(risk_report["lowVulnerabilityCount"], securityRiskLowThreshold, "Low")

    license_risk_high = Risk(risk_report["licensesLegalRisk"]["high"], licenseRiskHighThreshold, "High")
    license_risk_medium = Risk(risk_report["licensesLegalRisk"]["medium"], licenseRiskMediumThreshold, "Medium")
    license_risk_low = Risk(risk_report["licensesLegalRisk"]["low"], licenseRiskLowThreshold, "Low")

    security_risk_profile = RiskProfile(risk_high, risk_medium, risk_low)
    license_risk_profile = RiskProfile(license_risk_high, license_risk_medium, license_risk_low)

    print("Risk Report Id : " + str(risk_report["riskReportId"]) + "\n")
    print("**Security Risk**\n&nbsp;\n")
    status = "| Severity       |      Value       |  Threshold    |   Status  | \n"
    status += "|--------------- |:----------------:|:-------------:|:---------:| \n"
    status += risk_high.get_display_text() + "\n"
    status += risk_medium.get_display_text() + "\n"
    status += risk_low.get_display_text() + "\n"
    status += "\n&nbsp;\n"
    print(status)
    print("**License Risk**\n&nbsp;\n")
    status = "| Severity       |      Value       |  Threshold    |   Status  | \n"
    status += "|--------------- |:----------------:|:-------------:|:---------:| \n"
    status += license_risk_high.get_display_text() + "\n"
    status += license_risk_medium.get_display_text() + "\n"
    status += license_risk_low.get_display_text() + "\n"
    status += "\n&nbsp;\n"
    print(status)

    compliance_data = {}

    for security_risk in [risk_high, risk_medium, risk_low]:
        compliance_data["Security " + str(security_risk.get_type())] = {
            "Value": str(security_risk.get_value()),
            "Threshold": str(security_risk.get_threshold()),
            "Status": "Pass" if security_risk.validate() else "Fail"
        }

    for license_risk in [license_risk_high, license_risk_medium, license_risk_low]:
        compliance_data["License " + str(license_risk.get_type())] = {
            "Value": str(license_risk.get_value()),
            "Threshold": str(license_risk.get_threshold()),
            "Status": "Pass" if license_risk.validate() else "Fail"
        }
    compliance_data["riskReportId"] = str(risk_report["riskReportId"])
    analysis_date = risk_report["createdOn"]

    if security_risk_profile.validate() and license_risk_profile.validate():
        add_code_compliance_record(taskReportingApi, task, checkmarxScaServer, project=projectName, project_url="Not available",
                                  outcome="PASSED", compliance_data=compliance_data, analysis_date=analysis_date)
    else:
        add_comment("Issues count is higher than allowed threshold")
        add_code_compliance_record(taskReportingApi, task, checkmarxScaServer, project=projectName, project_url="Not available",
                                  outcome="FAILED", compliance_data=compliance_data, analysis_date=analysis_date)
        sys.exit(1)

except Exception as e:
    add_comment("Exception occurred: %s " % e)
    sys.exit(1)
