package com.xebialabs.xlrelease.audit


import com.google.common.base.Strings
import com.xebialabs.deployit.audit.CiFormatter
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.events.XLReleaseEvent
import com.xebialabs.xlrelease.events.EventListener
import grizzled.slf4j.Logging
import org.slf4j.LoggerFactory


object Messages {

  implicit class ConfigurationItemUtils(val ci: ConfigurationItem) extends AnyVal {
    def format = s"${ci.getType}[${ci.getId}]"
  }

  def created(ci: ConfigurationItem) = s"Created CI ${ci.format}"

  def updated(ci: ConfigurationItem) = s"Updated CI ${ci.format}"

  def updated(cis: Seq[ConfigurationItem]) = s"Updated CIs: ${cis.map(_.format).mkString(", ")}"

  def moved(ci: ConfigurationItem, oldId: String) = s"Moved CI ${ci.format} from $oldId"

  def copied(ci: ConfigurationItem) = s"Copied CI ${ci.format}"

  def copied(ci: ConfigurationItem, newId: String) = s"Copied CI ${ci.format} to new id $newId"

  def deleted(ci: ConfigurationItem) = s"Deleted CI ${ci.format}"

  def renamed(ci: ConfigurationItem, newName: String) = s"Renamed CI ${ci.format} to $newName"
}

object Components {
  val repository = "repository"
  val system = "system"
  val security = "security"
}

@EventListener
trait AuditLogger[T <: XLReleaseEvent] extends Logging {

  val formatter = new CiFormatter

  private def ignore: PartialFunction[XLReleaseEvent, AuditParams] = {
    case _ => AuditParams.ignored
  }

  def log(event: XLReleaseEvent): Unit = {
    (audit orElse ignore).apply(event) match {
      case AuditParams(component, message, cis) if !Strings.isNullOrEmpty(component) =>
        val auditor = LoggerFactory.getLogger("audit." + component)
        if (auditor.isInfoEnabled) {
          auditor.info(s"${event.username} - $message")
          cis.foreach(ci =>
            auditor.info(s"${event.username} - CI [${ci.getId}]:\n${formatter.format(ci)}")
          )
        }
      case _ => logger.debug(s"Event $event not traced into audit log")
    }
  }

  def audit: PartialFunction[XLReleaseEvent, AuditParams]

}
