package com.xebialabs.xlrelease.audit

import com.xebialabs.xlrelease.audit.Components.security
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import org.springframework.stereotype.Component

@Component
@EventListener
class AccountLockAuditHandler extends AuditLogger[AccountEvents] {

  @Subscribe
  def onAccountEvents(event: AccountEvents): Unit = log(event)

  override def audit: PartialFunction[XLReleaseEvent, AuditParams] = {
    case event @ AccountLockedEvent(failedCount, autoUnlockMinutes) =>
      AuditParams(security, s"Account locked for user [${event.user}] after reaching $failedCount failed attempts." +
        s"Auto-unlock in $autoUnlockMinutes minutes.")
    case event @ AccountUnlockAndFailedAttemptsResetEvent(failedCount) =>
      AuditParams(security, s"Account for user [${event.user}] unlocked after threshold period." +
        s"Failed attempt counter reset to $failedCount for this login failure.")
    case event @ FailedLoginAttemptEvent(failedCount, maxFailedAttempts) =>
      AuditParams(security, s"Failed login attempt [$failedCount] recorded for user [${event.user}]." +
        s"Account will lock after ${maxFailedAttempts - failedCount} more failed attempts.")
    case event @ AutoLockThresholdNotReachedEvent(failedCount, minutesRemaining) =>
      AuditParams(security, s"Account for user [${event.user}] is still locked (count=$failedCount)." +
        s"Auto-unlock in $minutesRemaining minutes.")
    case event @ AccountUnlockedAfterSuccessfulLoginEvent() =>
      AuditParams(security, s"Successful login: failed attempts counter reset and account for user [${event.user}] unlocked if previously locked.")
    case event @ AuthenticationFailureLockEvent() =>
      AuditParams(security, s"Authentication failure: Account locked for user: [${event.user}]")
    case event @ AccountUnlockEvent() =>
      AuditParams(security, s"Account for user [${event.user}] unlocked by ${event.username}.")
  }
}
