#
# Copyright (c) 2021. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#
import json
import urllib
from xlrelease.HttpRequest import HttpRequest

from com.xebialabs.xlrelease.plugin.argocd import LoaderUtil


class RestClient:

    def __init__(self, argo_server=None):
        argo_server['authenticationMethod'] = 'None'
        self.httpRequest  = HttpRequest(argo_server, verify=False)
        self.username = argo_server['username']
        self.password = argo_server['password']
        self.auth_token = argo_server['authToken']

    def getBearerToken(self):
        if self.auth_token is not None and self.auth_token != "":
            print "The provided authToken can be used as a bearer token"
            return self.auth_token
        # Create POST body
        content = {
            'username': self.username,
            'password': self.password
        }
        response = self.httpRequest.post('/api/v1/session', json.dumps(content),
                                contentType='application/json', headers=None)
        if response.isSuccessful():
            response_json = json.loads(response.getResponse())
            bearer_token = response_json['token']
        else:
            raise ValueError('Failed to get bearer token %s:%s' % (response.getStatus(),response.getResponse()))
        return bearer_token

    def checkConnection(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/projects', contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to connect the server %s:%s' % (response.getStatus(), response.getResponse()))

    def listClusters(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/clusters', contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to list clusters %s:%s' % (response.getStatus(), response.getResponse()))

        responseObj = json.loads(response.getResponse())
        clusters = responseObj["items"]
        return clusters

    def listProjects(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/projects', contentType='application/json', headers=headers)
        result = {}
        projects = []
        if response.isSuccessful():
            responseObj = json.loads(response.getResponse())
            items = responseObj.get("items") if responseObj.get("items") is not None else []
            for item in items:
                project = {}
                spec = item["spec"]
                metadata = item["metadata"]
                project["name"] = metadata.get("name")
                project["namespace"] = metadata.get("namespace")
                project["description"] = spec.get("description")
                project["sourceRepos"] = spec.get("sourceRepos")
                project["destinations"] = spec.get("destinations")
                projects.append(project)
        else:
            raise ValueError('Failed to get project list %s:%s' % (response.getStatus(),response.getResponse()))
        result["projects"] = projects
        return result

    def getRepoConnections(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/repositories', contentType='application/json', headers=headers)
        repos = None
        if response.isSuccessful():
            repositories = json.loads(response.getResponse())
        else:
            raise ValueError('Failed to get repositories list %s:%s' % (response.getStatus(),response.getResponse()))
        return repositories

    def listRepositories(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        repos = None
        repositories = self.getRepoConnections()
        repos = []
        items = repositories.get("items") if repositories.get("items") is not None else []
        for item in items:
            repo = item.get("repo")
            if repo is not None :
                repos.append(repo)

        return repos

    def listApplications(self, projects):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        params = ''
        for project in projects:
            params += '&projects=' + project + '&project=' + project
        response = self.httpRequest.get('/api/v1/applications?' + params, contentType='application/json', headers=headers)
        applications = []
        if response.isSuccessful():
            responseObj = json.loads(response.getResponse())
            items = responseObj.get("items") if responseObj.get("items") is not None else []
            for item in items:
                metadata = item.get("metadata", {})
                if metadata.get("deletionTimestamp") is None:
                    spec = item.get("spec", {})
                    status = item.get("status", {})
                    application = {}
                    application["name"] = metadata.get("name")
                    application["path"] = "/applications/" + metadata.get("name") + "?view=tree"
                    application["created_at"] = metadata.get("creationTimestamp")
                    application["namespace"] = metadata.get("namespace", {})
                    application["labels"] = metadata.get("labels", {})
                    application["destination"] = spec.get("destination", {})
                    application["status"] = status

                    repo_url = spec.get("source", {}).get("repoURL")
                    application["repo_url"] = repo_url

                    revision = status.get("sync", {}).get("revision")
                    application["revision"] = revision
                    revision_metadata = self.get_revision_metadata(application["name"], revision) if revision is not None else {}
                    application["revision_metadata"] = revision_metadata if revision_metadata is not None else None

                    last_change = status.get("operationState", {}).get("finishedAt")
                    application["last_change"] = last_change if last_change is not None else metadata.get("creationTimestamp")

                    applications.append(application)
        else:
            raise ValueError('Failed to get application list %s:%s' % (response.getStatus(), response.getResponse()))
        return applications

    def addRepository(self, url, repoType='git', repoName=None, project=None, username=None, password=None, sshKey=None,
                      tlsClientCertData=None, tlsClientCertKey=None, upsert=False):

        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        body = {
            "type": repoType,
            "repo": url
        }
        if repoName:
            body["name"] = repoName
        if project:
            body["project"] = project
        if username:
            body["username"] = username
        if password:
            body["password"] = password
        if sshKey:
            body["sshPrivateKey"] = sshKey

        if sshKey:
            body["tlsClientCertData"] = tlsClientCertData
        if sshKey:
            body["tlsClientCertKey"] = tlsClientCertKey

        response = self.httpRequest.post("/api/v1/repositories?upsert=%s" % (upsert), json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
           raise ValueError('Failed to add repository list %s:%s' % (response.getStatus(),response.getResponse()))

    def syncApplication(self, appName, autoCreateNamespace, refresh, pruneResources, prunePropagationPolicy):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}

        items = []
        body = {}
        if autoCreateNamespace:
            items.append("CreateNamespace=true")
        if pruneResources:
            body = {"prune": True}
            items.append("PrunePropagationPolicy=%s" % prunePropagationPolicy)
        sync_options = {'items': items}
        body["syncOptions"] = sync_options

        if refresh:
            refresh_endpoint = "/api/v1/applications/" + appName + "?refresh=%s" % refresh
            refresh_response = self.httpRequest.post(refresh_endpoint, json.dumps(body), contentType='application/json', headers=headers)
            if not refresh_response.isSuccessful:
                raise ValueError('Failed to refresh app %s:%s' % (refresh_response.getStatus(),refresh_response.getResponse()))

        endpoint = "/api/v1/applications/" + appName + "/sync"
        response = self.httpRequest.post(endpoint, json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to sync app list %s:%s' % (response.getStatus(),response.getResponse()))
        else:
            responseObj = json.loads(response.getResponse())
            statusObj = responseObj["status"]
            syncObj = statusObj["sync"]
            status = syncObj["status"]
            if "revision" in syncObj:
                revision = syncObj['revision']
            else:
                revision = None
            result = {
                "status" : status,
                "revision": revision,
            }
        return result

    def getSyncStatus(self, appName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName

        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)
        if not response.isSuccessful():
            raise ValueError('Failed to get sync status %s:%s' % (response.getStatus(),response.getResponse()))

        deployment_status = {}

        statusObj = None
        revision = None
        syncStatus = None
        healthStatus = None
        conditionsObj = None
        responseObj = json.loads(response.getResponse())

        if "status" in responseObj:
            statusObj = responseObj["status"]
            if "sync" in statusObj:
                sync = statusObj["sync"]
                if "revision" in sync:
                    revision = sync["revision"]
                if "status" in sync:
                    syncStatus = sync["status"]
                    if syncStatus == "Unknown":
                        conditionsObj = statusObj["conditions"]
            if "health" in statusObj:
                healthObj = statusObj["health"]
                if "status" in healthObj:
                    healthStatus = healthObj["status"]

        syncFailed = False
        message = ""
        if "operationState" in statusObj :
            opState = statusObj["operationState"]
            if opState["phase"] not in ["Synced", "Succeeded"]:
                if "message" in opState:
                    message = opState["message"]
                    if "successfully synced" not in message:
                        syncFailed = True
        deployment_status = {
            "syncStatus": syncStatus,
            "healthStatus": healthStatus
        }
        result = {
            "status" : deployment_status,
            "revision" : revision,
            "syncFailed" : syncFailed,
            "message" : message,
            "conditions": conditionsObj
        }

        return result

    def createProject(self, projectName, sourceRepos, destinations):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        body = { "upsert": True }
        project = {
            "metadata": {
                "name": projectName
            }
        }
        spec = {}
        if sourceRepos:
            sourceRepoList = []
            for repo in sourceRepos:
                sourceRepoList.append(repo)
            spec["sourceRepos"] = sourceRepoList

        if destinations:
            destinationList = []
            for serverKey in destinations:
                destObj =  {
                    "namespace": destinations[serverKey],
                    "server": serverKey
                }
                destinationList.append(destObj)
            spec["destinations"] = destinationList

        clusterResourceWhitelist = [{
            "group": "*",
            "kind": "*"}]

        spec["clusterResourceWhitelist"] = clusterResourceWhitelist

        project["spec"] = spec
        body["project"] = project

        response = self.httpRequest.post('/api/v1/projects', json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to create project %s:%s' % (response.getStatus(),response.getResponse()))

        return response.getResponse()

    def getProject(self, project_name):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/projects/" + project_name
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get project details %s:%s' % (response.getStatus(),response.getResponse()))

        return json.loads(response.getResponse())

    def getAppDetails(self, appName, bearer_token):
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get application details %s:%s' % (response.getStatus(),response.getResponse()))

        return json.loads(response.getResponse())

    def getResourceDetails(self, appName, resourceName, bearer_token):
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName + "/managed-resources?name="+resourceName
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get resource details %s:%s' % (response.getStatus(),response.getResponse()))

        return json.loads(response.getResponse())

    def getResourceTree(self, appName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName+"/resource-tree"
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get sync status %s:%s' % (response.getStatus(), response.getResponse()))

        return json.loads(response.getResponse())

    def get_revision_metadata(self, appName, revision):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/%s/revisions/%s/metadata" % (appName, revision)
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)
        if not response.isSuccessful():
            raise ValueError('Failed to get revision metadata %s:%s' % (response.getStatus(), response.getResponse()))

        return json.loads(response.getResponse())

    def add_notification_annotations(self, projects):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        params = ''
        for project in projects:
            params += '&projects=' + project + '&project=' + project
        response = self.httpRequest.get('/api/v1/applications?' + params, contentType='application/json', headers=headers)

        if response.isSuccessful():
            response_obj = json.loads(response.getResponse())
            items = response_obj.get("items") if response_obj.get("items") is not None else []
            for item in items:
                app_name = item.get("metadata", {}).get("name")
                item['metadata']['annotations'] = item.get("metadata", {}).get('annotations', {})
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-created.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-deleted.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-unknown.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-progressing.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-suspended.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-healthy.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-degraded.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-missing.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-sync-unknown.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-sync-synced.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-sync-out-of-sync.release-webhook'] = ''

                update_annotations_response = self.httpRequest.put('/api/v1/applications/' + app_name, json.dumps(item), contentType='application/json', headers=headers)

                if not update_annotations_response.isSuccessful():
                    raise ValueError('Failed to update application %s:%s' % (response.getStatus(),response.getResponse()))

            return items
        else:
            raise ValueError('Failed to get application list %s:%s' % (response.getStatus(), response.getResponse()))

    @staticmethod
    def generateYaml(url, token):
        res = LoaderUtil.getResourceBySelfClassLoader("argocd/templates/configmap.yaml")
        file_data = ""

        if res:
            with open(res.openStream()) as file :
               for line in file:
                   if token == "" and (line.find("Authorization") != -1 or line.find("Bearer $TOKEN") != -1):
                       pass
                   else:
                       file_data += line

        file_data = file_data.replace("$XL_RELEASE_URL", url)
        file_data = file_data.replace("$TOKEN", token)

        return file_data


    def createApplication(self, appName, project, server, namespace, url,
            repoType, path, chartName=None, revision=None, helmFileParameters=None,
            ignoreMissingValueFiles=False, parameters=None, passCredentials=False,
            releaseName=None, skipCrds=False, valueFiles=None, values=None,
            helmTemplatingVersion=None, upsert=False, validate=False, autoSync=False,
            pruneResources=False, selfHeal=False, skipValidation=False, pruneLast=False,
            autoCreateNamespace=False, applyOutOfSyncOnly=False, prunePropagationPolicy=None,
            replace=False, kustomizeCommonAnnotations=None, commonAnnotationsEnvsubst=False,
            kustomizeCommonLabels=None, forceCommonAnnotations=False, forceCommonLabels=False,
            kustomizeImages=None, kustomizeNamePrefix=None, kustomizeNameSuffix=None,
            kustomizeNamespace=None, kustomizeVersion=None, forceHelmParameters=False):

        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        body = {
            "metadata": {
                "name": appName
            },
            "spec": {
                "destination": {
                    "namespace": namespace,
                    "server": server
                },
                "project": project
            }
        }

        source = {
            "repoURL": url
        }

        if repoType == "git":
            source["path"] = path
        elif repoType == "helm":
            source["chart"] = chartName

        if revision:
            source["targetRevision"] = revision

        #Helm options
        if repoType == "helm" or forceHelmParameters:
            helmOptions={}
            fileParams = helmFileParameters
            if fileParams:
                helmFileParameters = []
                for paramName in fileParams:
                    helmFileParameters.append({'name': paramName, 'path': fileParams[paramName]})
                helmOptions['FileParameters'] = helmFileParameters

            helmOptions['ignoreMissingValueFiles'] = ignoreMissingValueFiles

            if parameters:
                helmParameters = []
                for paramName in parameters:
                    helmParameters.append({'name': paramName, 'value': parameters[paramName]})
                helmOptions['parameters'] = helmParameters

            helmOptions['passCredentials'] = passCredentials
            helmOptions['skipCrds'] = skipCrds
            if releaseName:
                helmOptions['releaseName'] = releaseName
            if valueFiles:
                helmOptions['valueFiles'] = valueFiles
            if values:
                helmOptions['values'] = values
            if helmTemplatingVersion:
                helmOptions['version'] = helmTemplatingVersion

            source["helm"] = helmOptions

        #Kustomize Options
        if repoType == "git":
            kustomizeOptions = {}
            if kustomizeCommonAnnotations:
                kustomizeOptions['commonAnnotations'] = kustomizeCommonAnnotations
                kustomizeOptions['commonAnnotationsEnvsubst'] = commonAnnotationsEnvsubst
                kustomizeOptions['forceCommonAnnotations'] = forceCommonAnnotations
            if kustomizeCommonLabels:
                kustomizeOptions['commonLabels'] = kustomizeCommonLabels
                kustomizeOptions['forceCommonLabels'] = forceCommonLabels
            if kustomizeImages:
                kustomizeOptions['images'] = kustomizeImages
            if kustomizeNamePrefix:
                kustomizeOptions['namePrefix'] = kustomizeNamePrefix
            if kustomizeNameSuffix:
                kustomizeOptions['nameSuffix'] = kustomizeNameSuffix
            if kustomizeNamespace:
                kustomizeOptions['namespace'] = kustomizeNamespace
            if kustomizeVersion:
                kustomizeOptions['version'] = kustomizeVersion

            if kustomizeOptions:
                source["kustomize"] = kustomizeOptions

        body["spec"]["source"] = source

        #Sync Options
        syncPolicy = {}
        if autoSync:
            automated = {}
            automated["prune"] = pruneResources
            automated["selfHeal"] = selfHeal
            syncPolicy["automated"] = automated

        syncOptions = []
        syncOptions.append("Validate=%s" % (not skipValidation))
        syncOptions.append("PruneLast=%s" % (pruneLast))
        syncOptions.append("CreateNamespace=%s" % (autoCreateNamespace))
        syncOptions.append("ApplyOutOfSyncOnly=%s" % (applyOutOfSyncOnly))
        syncOptions.append("Replace=%s" % (replace))
        syncOptions.append("PrunePropagationPolicy=%s" % (prunePropagationPolicy))
        syncPolicy["syncOptions"] = syncOptions

        body["spec"]["syncPolicy"] = syncPolicy

        url = "/api/v1/applications?validate=%s&upsert=%s" % (validate,upsert)

        response = self.httpRequest.post(url, json.dumps(body), contentType='application/json', headers=headers)
        if not response.isSuccessful():
            raise ValueError('Failed to create app %s:%s' % (response.getStatus(),response.getResponse()))

        return response.getResponse()

    def getChartVersions(self, repo_url):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "api/v1/repositories/{repo}/helmcharts".format(repo=urllib.quote_plus(repo_url))
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to list helm charts versions %s:%s' % (response.getStatus(),response.getResponse()))

        return json.loads(response.getResponse())

    def getRevisions(self, repo_url):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "api/v1/repositories/{repo}/refs".format(repo=urllib.quote_plus(repo_url))
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to list revisions %s:%s' % (response.getStatus(),response.getResponse()))

        return json.loads(response.getResponse())

    def listAppNames(self, projectName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications"
        if projectName:
            endpoint += "?project=%s" % projectName
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)
        app_names = []
        if response.isSuccessful():
            responseObj = json.loads(response.getResponse())
            items = responseObj.get("items") if responseObj.get("items") is not None else []
            for item in items:
                metadata = item.get("metadata", {})
                if metadata.get("deletionTimestamp") is None:
                    app_names.append(metadata.get("name"))
        else:
            raise ValueError('Failed to get application list %s:%s' % (response.getStatus(), response.getResponse()))
        return app_names

    def listAppHistoryIdsAndRevisions(self, appName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/{app}".format(app=appName)
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)
        if not response.isSuccessful():
            raise ValueError('Failed to list ids %s: %s' % (response.getStatus(),response.getResponse()))
        data = json.loads(response.getResponse())
        history = data.get("status", {}).get("history", [])
        app_history_ids_and_revisions = [
            "{id}-{revision}".format(id=str(entry['id']), revision=str(entry['revision'][:7]))
            for entry in history
        ]

        return app_history_ids_and_revisions

    def getAppHistory(self, appName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/{app}".format(app=appName)
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)
        data = json.loads(response.getResponse())
        history = data.get("status", {}).get("history", [])
        status  = response.getStatus()
        result = {
        "history" : history,
        "status" : status
        }
        return result

    def rollbackApplication(self, historyid, appName, appNamespace, pruneResources, dryRun):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/{app}/rollback".format(app=appName)
        getHistory = self.getAppHistory(appName)
        history = getHistory["history"]
        if len(history) < 2:
            raise ValueError('No previous version to rollback to')
        body = {"prune": pruneResources} if pruneResources else {}
        # Previous history ID
        if historyid is None:
            body["id"] = int(history[-2]['id'])
        else:
            body["id"] = int(historyid.split("-")[0])
        if appNamespace:
            body["appNamespace"] = appNamespace
        if dryRun:
            body["dryRun"] = dryRun

        response = self.httpRequest.post(endpoint, json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError("Failed to rollback app '%s'. '%s' : %s" % (appName, response.getStatus(), response.getResponse()))

        return json.loads(response.getResponse())

    def deleteApplication(self, appName, propagationPolicy, appNamespace):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = ("/api/v1/applications/{appName}?PropagationPolicy={propagationPolicy}&cascade=true"
                    .format(appName=appName, propagationPolicy=propagationPolicy))

        if propagationPolicy == "non-cascading":
            endpoint = "/api/v1/applications/{appName}?PropagationPolicy=&cascade=false".format(appName=appName)
        if appNamespace:
            endpoint += "&namespace=%s" % appNamespace
        response = self.httpRequest.delete(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError("Failed to delete app '%s'. '%s': %s" % (appName, response.getStatus(), response.getResponse()))

    def deleteRepository(self, repoUrl):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/repositories/{repoUrl}".format(repoUrl=repoUrl)
        response = self.httpRequest.delete(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError("Failed to delete repository '%s'. '%s': %s" % (repoUrl, response.getStatus(), response.getResponse()))
        
    def deleteProject(self, projectName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/projects/{projectName}".format(projectName=projectName)
        response = self.httpRequest.delete(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError("Failed to delete project '%s'. '%s': %s" % (projectName, response.getStatus(), response.getResponse()))
