#
# Copyright (c) 2021. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#
from com.xebialabs.xlrelease.status.webhook.events import UpdateStatusEvent, StateMetadata, VersionState
from java.util import ArrayList, TimeZone
from java.text import SimpleDateFormat

class ObjectView(object):
    def __init__(self, d):
        self.__dict__ = d

    def getProperty(self, property):
        return self.__dict__[property]

def map_application_deployment_package_states(response, clusters):
    data = ArrayList()
    for app_status in response:
        if app_status.get("revision") is not None:
            data.add(map_application_deployment_package_state(app_status, clusters))
    return data

def map_status_group(status):
    if status:
        status = status.lower()
    if status == 'healthy':
        return 'success'
    elif status == 'suspended' or status == 'degraded' or status == 'missing' or status == 'unknown':
        return 'failed'
    else:
        return 'in-progress'

def map_application_deployment_package_state(app_status, clusters):
    name = app_status.get("name")
    path = app_status.get("path")
    namespace = app_status.get("destination", {}).get("namespace")
    application_uid = namespace + ":" + name
    status = app_status.get("status", {})
    status_group = map_status_group("Unknown" if has_no_value(status.get("health", {}).get("status")) else status.get("health", {}).get("status"))
    deployment = map_deployment_package_state(app_status, clusters)
    return UpdateStatusEvent(name, application_uid, path, status_group, deployment)


def map_deployment_package_state(app_status, clusters):
    cluster_url = app_status.get("destination", {}).get("server", "Unknown")
    destination = clusters.get(cluster_url, cluster_url)
    namespace = app_status.get("destination", {}).get("namespace")
    destination_uid = namespace + ":[" + cluster_url + "]"
    version = app_status.get("repo_url")[:-4] + "/commit/" + app_status.get("revision")
    status = app_status.get("status", {})
    sync_status = "Unknown" if has_no_value(status.get("sync", {}).get("status")) else status.get("sync", {}).get("status")
    health_status = "Unknown" if has_no_value(status.get("health", {}).get("status")) else status.get("health", {}).get("status")
    user = "Unknown" if has_no_value(app_status.get("revision_metadata", {}).get("author")) else app_status.get("revision_metadata", {}).get("author")

    sdf = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'")
    sdf.setTimeZone(TimeZone.getTimeZone("UTC"))
    change_time = None if has_no_value(app_status.get("last_change")) else sdf.parse(app_status.get("last_change"))

    return StateMetadata(
        destination,
        destination_uid,
        namespace,
        VersionState(version, sync_status),
        health_status,
        None,
        user,
        change_time
    )

def map_cluster_names(clusters, server_name):
    if clusters is None:
        raise Exception('There are no clusters defined on ArgoCD server %s', server_name)
    clusters_dict = {}
    for c in clusters:
        if c.get("server") is not None:
            clusters_dict[c.get("server")] = c.get("name")
    return clusters_dict

def has_no_value(value):
    return value is None or value == "<no value>"

def map_projects(response):
    data = ArrayList()
    for project in response.get("projects", []):
        data.add(project.get("name"))
    return data
