'use strict';

angular.module('xlrelease').factory('ReleasesService', ['Backend', 'TasksService', '$q', function (Backend, TasksService, $q) {

    function getSubTasks(task) {
        if (TasksService.isParallelGroup(task)) {
            return _.map(task.tasks, getSubTasks);
        } else {
            return [ task ];
        }
    }

    function getLeafTasks(release) {
        return _.flatten(_.map(release.phases, function (phase) {
            return _.map(phase.tasks, getSubTasks);
        }));
    }

    return {
        getLeafTasks: getLeafTasks,
        isReleaseCompleted: function (release) {
            return angular.isDefined(release) && (release.status === 'COMPLETED' || release.status === 'ABORTED');
        },
        isReleaseAborted: function (release) {
            return angular.isDefined(release) && release.status === 'ABORTED';
        },
        isReleasePlanned: function (release) {
            return angular.isDefined(release) && release.status === 'PLANNED';
        },
        isReleaseInProgress: function (release) {
            return angular.isDefined(release) && release.status === 'IN_PROGRESS';
        },
        isReleaseFailed: function (release) {
            return angular.isDefined(release) && release.status === 'FAILED';
        },
        isReleaseFailing: function (release) {
            return angular.isDefined(release) && release.status === 'FAILING';
        },
        isReleasePaused: function (release) {
            return angular.isDefined(release) && release.status === 'PAUSED';
        },
        startRelease: function (release) {
            return Backend.post('releases/' + release.id + '/start');
        },
        abortRelease: function (release) {
            return Backend.post('releases/' + release.id + '/abort');
        },
        downloadCalendar: function (release) {
            Backend.get('releases/' + release.id + '/calendar').success(function (downloadToken) {
                window.open('calendar/download/' + downloadToken);
            });
        },
        getReleasesDependencies: function (releases) {
            var params = _.map(releases, function (release) {
                return "releaseId=" + release.id;
            }).join("&");

            return Backend.get('dependencies?' + params)
        },
        getReleaseDependencies: function (releaseId) {
            return Backend.get('dependencies/' + releaseId)
        },
        getRelease: function (releaseId) {
            return Backend.get('releases/' + releaseId);
        },
        cropToReleaseDate: function (phase, release) {
            return {
                startDate: moment(phase.scheduledStartDate).isBefore(moment(release.scheduledStartDate)) ? moment(release.scheduledStartDate) : moment(phase.scheduledStartDate),
                dueDate: moment(phase.dueDate).isAfter(moment(release.dueDate)) ? moment(release.dueDate) : moment(phase.dueDate)
            };
        },
        getReleasesDependenciesByReleaseId: function (releases) {
            var deferred = $q.defer();
            var promise = deferred.promise;

            if (releases.length === 0) {
                deferred.reject();
                return promise;
            }

            this.getReleasesDependencies(releases).success(function (releasesDependencies) {
                deferred.resolve(_.indexBy(releasesDependencies, 'releaseId'));
            });

            return promise;
        },
        isInConflict: function (releaseId, releasesDependencies) {
            return angular.isDefined(releasesDependencies)
                && angular.isDefined(releasesDependencies[releaseId])
                && releasesDependencies[releaseId].inConflict;
        }
    };
}]);
