'use strict';

angular.module('xlrelease').controller('releaseEditorController', ['$scope', '$routeParams', 'Backend', 'TaskPolling', 'TasksService', 'PhaseColors', 'ReleasesService', 'Modal', 'FlagService', 'Authenticator', function ($scope, $routeParams, Backend, TaskPolling, TasksService, PhaseColors, ReleasesService, Modal, FlagService, Authenticator) {

    $scope.PhaseColors = PhaseColors;

    $scope.init = {
        hasScrolledToCurrentTask: false
    };

    $scope.loadRelease = function () {
        $scope.$emit('REFRESH_PERMISSIONS');
        Backend.get('releases/' + $routeParams.releaseId).success(function (release) {
            $scope.release = release;
            $scope.allTasks = ReleasesService.getLeafTasks($scope.release);
            $scope.hasAutomatedTaskInProgress = _.some($scope.allTasks, function (task) {
                return TasksService.isTaskInProgress(task) && TasksService.isAutomated(task);
            });
            initRestartPhases(release);
            startPolling();
        });
    };

    $scope.loadRelease();

    Backend.get('tasks/python-script-definitions').success(function (definitions) {
        $scope.pythonScriptDefinitions = _.groupBy(definitions, 'displayGroup');
    });

    function startPolling() {
        TaskPolling.startTaskPolling($scope.allTasks, $scope).onTaskUpdate($scope.loadRelease);
        Modal.withScope($scope).onNextOpen(stopPolling).onNextClose(startPolling)
    }

    function stopPolling() {
        $scope.$broadcast('STOP_TASK_POLLING');
    }

    $scope.isPhaseInProgress = function (phase) {
        return 'IN_PROGRESS' === phase.status;
    };

    $scope.isPhaseReadOnly = function (phase) {
        return 'COMPLETED' === phase.status || 'ABORTED' === phase.status || 'SKIPPED' === phase.status;
    };

    $scope.isPhasePlanned = function (phase) {
        return 'PLANNED' === phase.status;
    };

    $scope.startRelease = function (release) {
        ReleasesService.startRelease(release).success($scope.loadRelease);
    };

    $scope.abortRelease = function (release) {
        ReleasesService.abortRelease(release).success($scope.loadRelease);
    };

    $scope.showNavigator = false;

    $scope.movePhase = function (fromIndex, toIndex) {
        var moveInfo = {
            originIndex: fromIndex,
            targetIndex: toIndex
        };

        Backend.post('releases/' + $scope.release.id + '/phases/move', moveInfo).success(function (movedPhase) {
            $scope.release.phases.splice(fromIndex, 1);
            $scope.release.phases.splice(toIndex, 0, movedPhase);
        });
    };

    $scope.moveTask = function (fromTaskOrPhase, fromIndex, toTaskOrPhase, toIndex) {
        var moveInfo = {
            originContainerId: fromTaskOrPhase.id,
            originIndex: fromIndex,
            targetContainerId: toTaskOrPhase.id,
            targetIndex: toIndex
        };

        Backend.post('releases/' + $scope.release.id + '/tasks/move', moveInfo).success(function (movedTask) {
            fromTaskOrPhase.tasks.splice(fromIndex, 1);
            toTaskOrPhase.tasks.splice(toIndex, 0, movedTask);
        }).error(function (data) {
            $scope.moveTaskError = data;
        });
    };

    $scope.resetError = function () {
        $scope.moveTaskError = false;
    };

    $scope.addPhase = function () {
        Backend.post('releases/' + $scope.release.id + '/phases/add').success(function (phase) {
            $scope.release.phases.push(phase);
        });
    };

    $scope.deletePhase = function (phaseToDelete) {
        Backend.del('phases/' + phaseToDelete.id).success(function () {
            $scope.release.phases = _.filter($scope.release.phases, function (phase) {
                return phase.id !== phaseToDelete.id;
            })
        });
    };

    $scope.updatePhaseColor = function (phase, color) {
        phase.color = color;
        $scope.updatePhase(phase);
    };

    $scope.addTask = function (container, taskForm) {
        Backend.post('tasks/' + container.id, taskForm).success(function (addedTask) {
            container.tasks.push(addedTask);
        });
    };

    $scope.deleteTask = function (container, taskToDelete) {
        Backend.del('tasks/' + taskToDelete.id).success(function () {
            container.tasks = _.filter(container.tasks, function (task) {
               return task.id !== taskToDelete.id;
            });
        });
    };

    $scope.duplicateTask = function (container, task) {
        Backend.put('releases/' + $scope.release.id + '/tasks/duplicate/' + task.id ).success(function (duplicatedTask) {
            _.each(container.tasks, function (aTask, index) {
                if (aTask.id === task.id) {
                    container.tasks.splice(index + 1, 0, duplicatedTask);
                }
            });
        });
    };

    $scope.duplicatePhase = function (phase) {
        Backend.put('releases/' + $scope.release.id + '/phases/duplicate/' + phase.id ).success(function (duplicatedPhase) {
            _.each($scope.release.phases, function (aPhase, index) {
                if (aPhase.id === phase.id) {
                    $scope.release.phases.splice(index + 1, 0, duplicatedPhase);
                }
            });
        });
    };

    $scope.completeTask = function (task, commentText) {
        TasksService.completeTask(task.id, commentText).success($scope.loadRelease);
    };

    $scope.skipTask = function (task, commentText) {
        TasksService.skipTask(task.id, commentText).success($scope.loadRelease);
    };

    $scope.failTask = function (task, commentText) {
        TasksService.failTask(task.id, commentText).success($scope.loadRelease);
    };

    $scope.retryTask = function (task, commentText) {
        TasksService.retryTask(task.id, commentText).success($scope.loadRelease);
    };

    $scope.startNow = function (task, commentText) {
        TasksService.startNow(task.id, commentText).success($scope.loadRelease);
    };

    $scope.renameRelease = function () {
        var updateForm = {
            title: $scope.release.title
        };
        if ($scope.release.status === 'TEMPLATE') {
            Backend.put('releases/templates/' + $scope.release.id, updateForm);
        } else {
            Backend.put('releases/' + $scope.release.id, updateForm);
        }
    };

    function initRestartPhases(release) {
        $scope.restartPhases = {};
        $scope.restartPhases.restartables = _.reject(release.phases, $scope.isPhasePlanned);
        $scope.restartPhases.current = _.find($scope.restartPhases.restartables, isPhaseActive);
        $scope.restartPhases.from = $scope.restartPhases.current;
    }

    function isPhaseActive(phase) {
        return 'IN_PROGRESS' === phase.status || 'FAILED' === phase.status || 'FAILING' === phase.status;
    }

    $scope.canRestartPhases = function (release) {
        return release && release.status != 'TEMPLATE' && Authenticator.hasPermission('release#edit', release.id)
            && (ReleasesService.isReleaseInProgress(release) || ReleasesService.isReleaseFailed(release) || ReleasesService.isReleaseFailing(release))
            && !$scope.hasAutomatedTaskInProgress;
    };

    $scope.computePhasesToRestart = function () {
        $scope.restartPhases.toRestart = $scope.restartPhases.restartables.slice(
            _.indexOf($scope.restartPhases.restartables, $scope.restartPhases.from),
            _.indexOf($scope.restartPhases.restartables, $scope.restartPhases.current) + 1)
    };

    $scope.canResume = function (release) {
        return release && release.status != 'TEMPLATE'
            && Authenticator.hasPermission('release#edit', release.id)
            && ReleasesService.isReleasePaused(release);
    };

    $scope.resumeRelease = function (release, variables) {
        if (!variables) {
            variables = [];
        }
        Backend.post('releases/' + release.id + '/resume', variables).success($scope.loadRelease);
    };

    $scope.updatePhase = function (phase) {
        var phaseForm = {
            title: phase.title,
            description: phase.description,
            scheduledStartDate: phase.scheduledStartDate,
            dueDate: phase.dueDate,
            color: phase.color
        };
        Backend.put('phases/' + phase.id, phaseForm);
    };

    $scope.isTaskPlanned = TasksService.isTaskPlanned;
    $scope.isTaskPending = TasksService.isTaskPending;
    $scope.isTaskInProgress = TasksService.isTaskInProgress;
    $scope.isTaskReadOnly = TasksService.isTaskReadOnly;
    $scope.isTaskSkipped = TasksService.isTaskSkipped;
    $scope.isTaskFailed = TasksService.isTaskFailed;
    $scope.isDeployitTask = TasksService.isDeployitTask;
    $scope.isGateTask = TasksService.isGateTask;
    $scope.isTaskFlagged = FlagService.isTaskFlagged;
    $scope.isManualTask = TasksService.isManualTask;
    $scope.isNotificationTask = TasksService.isNotificationTask;
    $scope.isParallelGroup = TasksService.isParallelGroup;
    $scope.isTaskCompleted = TasksService.isTaskCompleted;
    $scope.isParallelGroupInProgress = TasksService.isParallelGroupInProgress;
    $scope.isAutomated = TasksService.isAutomated;
    $scope.isReleaseCompleted = ReleasesService.isReleaseCompleted;
    $scope.isReleasePlanned = ReleasesService.isReleasePlanned;
}]);

// Expose the container of a task, regardless if it's a phase or a group.
angular.module('xlrelease').controller('containerController', ['$scope', function ($scope) {
    $scope.$watch('release', function () {
        $scope.container = $scope.task || $scope.phase;
    });
}]);
