package com.xebialabs.xlrelease.webhooks.dispatcher

import com.xebialabs.xlplatform.webhooks.events.domain.{Event, EventConsumer, EventSource}
import com.xebialabs.xlplatform.webhooks.events.handlers.{EventConsumerHandler, EventDispatcher}
import com.xebialabs.xlrelease.webhooks.registry.SubscriberRegistry
import grizzled.slf4j.Logging
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component

import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.Try

/**
 * Dispatches events to event consumer handlers.
 *
 * @param subscriberRegistry
 * @param handlers
 */
@Component
class DefaultEventDispatcher(subscriberRegistry: SubscriberRegistry,
                             @Lazy
                             handlers: java.util.List[EventConsumerHandler[_, _]])
  extends EventDispatcher with Logging {

  private lazy val handlersScala: mutable.Buffer[EventConsumerHandler[Event, _ <: EventConsumer]] =
    handlers.asScala.map(_.asInstanceOf[EventConsumerHandler[Event, EventConsumer]])

  override def dispatch(event: Event): Unit = {
    logger.trace(s"dispatch($event) sourceId: ${event.sourceId}")
    for {
      (consumer, consumerHandler) <- consumers(event.sourceId)
      if consumer.consumerEnabled
    } {
      logger.trace(s" -> [$consumerHandler].consume($consumer, $event)")
      Try(consumerHandler.consume(consumer, event)).fold(
        t => logger.warn(s"Event consumer handler $consumerHandler failed to process $event: ${t.getMessage}", t),
        _ => ()
      )
    }
  }

  private def consumers(sourceId: EventSource.ID): Map[EventConsumer, _ <: EventConsumerHandler[Event, EventConsumer]] = {
    logger.info(s"lookup up consumers for $sourceId")
    val consumerEntries = for {
      consumerConfig <- subscriberRegistry.list(sourceId)
      consumer <- handlersScala.find(c => consumerConfig.getType.instanceOf(c.consumerConfigType)).orElse {
        logger.warn(s"no idea what to do with consumerConfig '$consumerConfig' of type '${consumerConfig.getType}'")
        None
      }
    } yield consumerConfig -> consumer.asInstanceOf[EventConsumerHandler[Event, EventConsumer]]
    consumerEntries.toMap
  }

}
