package com.xebialabs.xlrelease.webhooks.repository.persistence

import com.xebialabs.xlrelease.json.JsonUtils.objectMapper
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.xlplatform.webhooks.domain.Endpoint
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, FolderRepository, PersistenceInterceptor}
import com.xebialabs.xlrelease.webhooks.consumers.logging.SpyConsumer
import com.xebialabs.xlrelease.webhooks.registry.SubscriberRegistry
import com.xebialabs.xlrelease.webhooks.repository.persistence.WebhookEndpointInterceptor.CiEndpoint
import org.springframework.jdbc.core.JdbcTemplate

import scala.jdk.CollectionConverters._

object WebhookEndpointInterceptor {
  type CiEndpoint = BaseConfiguration with Endpoint
}

class WebhookEndpointInterceptor(configurationRepository: ConfigurationRepository,
                                 folderRepository: FolderRepository,
                                 val configurationPersistence: ConfigurationPersistence,
                                 val subscriberRegistry: SubscriberRegistry,
                                 val jdbcTemplate: JdbcTemplate,
                                 val dialect: Dialect)
  extends PersistenceInterceptor[BaseConfiguration]
    with EndpointPathPersistence {
  configurationRepository.registerPersistenceInterceptor(this)

  protected lazy val protectedConsumerTypes = Set(Type.valueOf(classOf[SpyConsumer]))
  private val CONFIGURATION_REFERENCE_TYPE = "Configuration"

  override def afterCreate(ci: BaseConfiguration): Unit = {
    ci match {
      case ci: CiEndpoint => insertEndpointPath(ci)
      case _ =>
    }
  }

  override def afterUpdate(ci: BaseConfiguration): Unit = {
    ci match {
      case ci: CiEndpoint => updateEndpointPath(ci)
      case _ =>
    }
  }

  override def onDelete(sourceId: String): Unit = {
    val consumers = subscriberRegistry.list(sourceId)
      .filterNot(consumer => protectedConsumerTypes.contains(consumer.getType))

    val confs = configurationRepository.findByIds[BaseConfiguration](consumers.map(_.getId).toList.asJava)
    val refs = confs.asScala.map(c => {
      val folderTitle = c.getFolderId match {
        case null => "Global"
        case "" => "Global"
        case _ => folderRepository.getTitle(c.getFolderId).getOrElse("")
      }
      ReferencedEntity(c.getId, c.getTitle, CONFIGURATION_REFERENCE_TYPE, c.getFolderId, folderTitle)
    })
    val json = objectMapper.writeValueAsString(refs.toList)
    if (consumers.nonEmpty) {
      throw new ItemInUseException(json)
    }
  }

}

private case class ReferencedEntity(id: String, title: String, referenceType: String, folderId: String, folderTitle: String)