package com.xebialabs.xlrelease.webhooks.authentication

import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlplatform.webhooks.authentication.RequestAuthenticationMethod
import com.xebialabs.xlplatform.webhooks.domain.Endpoint
import grizzled.slf4j.Logger

import scala.util.{Failure, Try}

object TokenAuthenticationMethod extends RequestAuthenticationMethod {
  val TOKEN_HEADER = "X-Release-Webhook-Token"

  // remove ugly '$' at the end of this singleton when logging
  lazy val logger: Logger = Logger(this.getClass.getName.dropRight(1))

  private lazy val passwordEncrypter: PasswordEncrypter = PasswordEncrypter.getInstance()

  override def authenticateScala(endpoint: Endpoint,
                                 headers: Map[String, String],
                                 params: Map[String, Array[String]],
                                 payload: String): Boolean = {
    val authentication: Try[Boolean] = for {
      token <- getTokenFromHeaders(headers)
      expectedToken <- getExpectedToken(endpoint.authentication.asInstanceOf[TokenAuthentication])
    } yield {
      token == expectedToken
    }

    authentication.recover {
      case err =>
        Option(err).foreach(e => logger.warn(e.getMessage, e))
        false
    }.get
  }

  private def getTokenFromHeaders(headers: Map[String, String]): Try[String] = {
    headers.find(k => k._1.equalsIgnoreCase(TOKEN_HEADER)) match {
      case Some(token) =>
        logger.trace(s"Got token from $TOKEN_HEADER header")
        Try(token._2)
      case None =>
        Failure(new IllegalArgumentException(s"Header '$TOKEN_HEADER' not found."))
    }
  }

  private def getExpectedToken(requestAuthenticationConfig: TokenAuthentication): Try[String] = {
    Option(requestAuthenticationConfig.webhookToken).collect {
      case webhookToken if webhookToken.nonEmpty =>
        Try(passwordEncrypter.ensureDecrypted(webhookToken))
    }.getOrElse {
      Failure(new IllegalStateException(s"'$requestAuthenticationConfig' has no webhookToken configured!"))
    }
  }

}
