package com.xebialabs.xlrelease.webhooks.service

import com.fasterxml.jackson.core.`type`.TypeReference
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.platform.script.jython.ScriptSource
import com.xebialabs.xlplatform.webhooks.events.domain.{DeploymentServerEvent, Event}
import com.xebialabs.xlrelease.domain.environments.LiveDeploymentConfig
import com.xebialabs.xlrelease.json.JsonUtils.objectMapper
import com.xebialabs.xlrelease.runner.impl.RunnerScriptService
import com.xebialabs.xlrelease.script.XlrScriptContext
import com.xebialabs.xlrelease.script.jython.{JythonScriptService, XlrJythonSupport}
import com.xebialabs.xlrelease.webhooks.consumers.StatusWebhookEventSource
import com.xebialabs.xlrelease.webhooks.service.script.CleanupScriptContextBuilder.createCleanupScriptContext
import com.xebialabs.xlrelease.webhooks.service.script.PatchScriptContextBuilder.createPatchScriptContext
import com.xebialabs.xlrelease.webhooks.service.script.StatusScriptContextBuilder.createStatusScriptContext
import com.xebialabs.xlrelease.webhooks.service.script.StatusWebhookAutoconfigureExecutor.createAutoconfigScriptContext
import com.xebialabs.xlrelease.webhooks.service.script.{AutoconfigResult, buildDefaultInputParams, ciAsMap}
import com.xebialabs.xlrelease.webhooks.utils.JythonScriptEventProcessingUtils
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service
import org.springframework.util.ResourceUtils

import java.io.FileNotFoundException
import scala.util.{Try, Using}

trait EventSourceOperationService {
  def executeStatusScript(eventSource: StatusWebhookEventSource): AnyRef
  def executeAutoconfigureScript(eventSource: StatusWebhookEventSource): AnyRef
  def executePatchScript(eventSource: StatusWebhookEventSource): AnyRef
  def executeCleanUpScript(eventSource: StatusWebhookEventSource, existingLiveDeploymentConfigs: Set[LiveDeploymentConfig]): AnyRef
  def executeFilterScript(eventSource: StatusWebhookEventSource, event: Event): Boolean
  def executeMapScript(eventSource: StatusWebhookEventSource, event: Event): Event
}

@Service
class DefaultEventSourceOperationService(val scriptService: JythonScriptService,
                                         jythonScriptUtils: JythonScriptEventProcessingUtils,
                                         runnerScriptService: RunnerScriptService) extends EventSourceOperationService with XlrJythonSupport with Logging {

  import jythonScriptUtils._
  private def executeScriptBasedOnRemote[T](eventSource: StatusWebhookEventSource,
                                            remoteExecutor: => T,
                                            localExecutor: => T): T = {
    if (eventSource.isRemote) remoteExecutor else localExecutor
  }

  override def executeStatusScript(eventSource: StatusWebhookEventSource): AnyRef =
    executeScriptBasedOnRemote[AnyRef](
      eventSource,
      executeStatusRemoteScript(eventSource),
      executeStatusJythonScript(eventSource)
    )

  override def executeAutoconfigureScript(eventSource: StatusWebhookEventSource): AnyRef =
    executeScriptBasedOnRemote(
      eventSource,
      executeAutoconfigureRemoteScript(eventSource),
      executeAutoconfigureJythonScript(eventSource)
    )

  override def executePatchScript(eventSource: StatusWebhookEventSource): AnyRef =
    executeScriptBasedOnRemote(
      eventSource,
      executePatchRemoteScript(eventSource),
      executePatchJythonScript(eventSource)
    )

  override def executeCleanUpScript(eventSource: StatusWebhookEventSource, existingLiveDeploymentConfigs: Set[LiveDeploymentConfig]): AnyRef =
    executeScriptBasedOnRemote(
      eventSource,
      executeCleanUpRemoteScript(eventSource, existingLiveDeploymentConfigs),
      executeCleanUpJythonScript(eventSource)
    )

  override def executeFilterScript(eventSource: StatusWebhookEventSource, event: Event): Boolean =
    executeScriptBasedOnRemote(
      eventSource,
      executeFilterRemoteScript(eventSource, event),
      executeFilterJythonScript(eventSource, event)
    )

  override def executeMapScript(eventSource: StatusWebhookEventSource, event: Event): Event =
    executeScriptBasedOnRemote(
      eventSource,
      executeMapRemoteScript(eventSource, event),
      executeMapJythonScript(eventSource, event)
    )

  private def executeStatusJythonScript(eventSource: StatusWebhookEventSource): AnyRef =
    execute(createStatusScriptContext(eventSource))

  private def executeFilterJythonScript(eventSource: StatusWebhookEventSource, event: Event): Boolean =
    nonEmptyScriptLocation(eventSource.filterScript).forall { path =>
      Try(ResourceUtils.getFile(path)).map { _ =>
        executeFilter(eventSource, event, sandbox = false)
          .apply(logger)
          .apply(ScriptSource.byResource(path))
      }.recover {
        case e: FileNotFoundException =>
          logger.error(s"Filter is missing script file, returning false, filter script has to be defined: ${e.getMessage}")
          false
        case e: Throwable =>
          logger.error("Filter caused exception, assuming false", e)
          false
      }.get
    }

  private def executeMapJythonScript(eventSource: StatusWebhookEventSource, event: Event): Event =
    executeMap(eventSource, event, sandbox = false)
      .apply(logger)
      .apply(ScriptSource.byResource(scriptLocationOrDefault(eventSource.getType, eventSource.mapScript)))

  private def executeAutoconfigureJythonScript(eventSource: StatusWebhookEventSource): AnyRef = {
    execute(createAutoconfigScriptContext(eventSource))
  }

  private def executePatchJythonScript(eventSource: StatusWebhookEventSource): AnyRef =
    execute(createPatchScriptContext(eventSource))

  private def executeCleanUpJythonScript(eventSource: StatusWebhookEventSource): AnyRef =
    execute(createCleanupScriptContext(eventSource))



  private def executeFilterRemoteScript(eventSource: StatusWebhookEventSource, event: Event): Boolean = {
    logger.debug(s"Executing filter script for StatusWebhookEventSource ${eventSource.getName}")
    val inputParams = buildDefaultInputParams(eventSource)
    inputParams.put("content", event.getProperty("content"))
    runnerScriptService.executeScript[Boolean](
      eventSource.filterScript,
      inputParams,
      new TypeReference[Boolean] {}
    )
  }

  private def executeMapRemoteScript(eventSource: StatusWebhookEventSource, event: Event): Event = {
    logger.debug(s"Executing map script for StatusWebhookEventSource ${eventSource.getName}")

    val inputParams = buildDefaultInputParams(eventSource)
    inputParams.put("content", event.getProperty("content"))
    runnerScriptService.executeScriptForObject[DeploymentServerEvent](
      eventSource.mapScript,
      inputParams)
  }

  private def executeAutoconfigureRemoteScript(eventSource: StatusWebhookEventSource): AnyRef = {
    val inputParams = buildDefaultInputParams(eventSource)
    inputParams.put("releaseUrl", ServerConfiguration.getInstance().getServerUrl)
    runnerScriptService.executeScript[AutoconfigResult](
      eventSource.autoconfigScript,
      inputParams,
      new TypeReference[AutoconfigResult] {}
    )
  }

  private def executePatchRemoteScript(eventSource: StatusWebhookEventSource): AnyRef = {
    runnerScriptService.executeScript[AnyRef](
      eventSource.patchScript,
      buildDefaultInputParams(eventSource),
      new TypeReference[AnyRef] {}
    )
  }

  private def executeCleanUpRemoteScript(eventSource: StatusWebhookEventSource, existingLiveDeploymentConfigs: Set[LiveDeploymentConfig]): AnyRef = {
    val inputParams = buildDefaultInputParams(eventSource)
    inputParams.put("existingLiveDeploymentConfigs", objectMapper.writeValueAsString(existingLiveDeploymentConfigs.map(ciAsMap)))
    runnerScriptService.executeScript[AnyRef](
      eventSource.cleanupScript,
      inputParams,
      new TypeReference[AnyRef] {}
    )
  }

  private def executeStatusRemoteScript(eventSource: StatusWebhookEventSource): AnyRef = {
    logger.debug(s"Executing status script for StatusWebhookEventSource ${eventSource.getName}")
    runnerScriptService.executeScript[DeploymentServerEvent](
      eventSource.statusScript,
      buildDefaultInputParams(eventSource))
  }

  private def execute(context: XlrScriptContext): AnyRef = {
    Using.resource(context) { scriptContext =>
      executeScript(scriptContext)
      scriptContext.getAttribute("data")
    }
  }
}

