package com.xebialabs.xlrelease.webhooks.mapping

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.{Metadata, Property}
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.webhooks.mapping.MappedProperty.{ConstantValue, PropertyValue, VariableValue}

import java.util.Date
import scala.beans.BeanProperty
import scala.util.matching.Regex


@Metadata(virtual = true)
sealed abstract class MappedProperty extends BaseConfigurationItem {

  @BeanProperty
  @Property
  var targetProperty: String = _

  protected def showSource: String

  override def toString: String =
    s"MappedProperty.${this.getClass.getSimpleName}($targetProperty <- $showSource)"

  def fold[A](onConstantValue: => ConstantValue[Any] => A,
              onVariableValue: => VariableValue => A,
              onPropertyValue: => PropertyValue => A): A = this match {
    case constant: ConstantValue[Any@unchecked] =>
      onConstantValue(constant)
    case variable: VariableValue =>
      onVariableValue(variable)
    case property: PropertyValue =>
      onPropertyValue(property)
  }

  def formattedProperty: String = targetProperty match {
    case MappedProperty.variablesRegex(variableName) => variableName
    case _ => targetProperty
  }

  def formattedValue: Any = showSource
}

object MappedProperty {

  val variablesRegex: Regex = """variables\[(.*)\]""".r

  def constant[Value <: Any](target: String, value: Value): ConstantValue[Value] = ConstantValue[Value](target, value)

  def variable(target: String, key: String): VariableValue = VariableValue(target, key)

  def property(target: String, source: String): PropertyValue = PropertyValue(target, source)

  def password(target: String, value: String): PasswordValue = PasswordValue(target, value)

  @Metadata(virtual = false)
  class VariableValue(target: String) extends MappedProperty {
    def this() = this(null)

    targetProperty = target

    @BeanProperty
    @Property
    var variableKey: String = _

    protected def showSource: String = s"$${${variableKey}}"
  }

  object VariableValue {
    def apply(target: String, variableKey: String): VariableValue = {
      val m = new VariableValue(target)
      m.variableKey = variableKey
      m
    }
  }


  @Metadata(virtual = false)
  class PropertyValue(target: String) extends MappedProperty {
    def this() = this(null)

    targetProperty = target

    @BeanProperty
    @Property
    var sourceProperty: String = _

    protected def showSource: String = sourceProperty
  }

  object PropertyValue {
    def apply(target: String, source: String): PropertyValue = {
      val m = new PropertyValue(target)
      m.sourceProperty = source
      m
    }
  }

  sealed abstract class ConstantValue[Value <: Any](kind: PropertyKind, target: String) extends MappedProperty {
    def this() = this(null, null)

    targetProperty = target

    def value: Value

    protected def showSource: String = String.valueOf(value)

    override def formattedValue: Any = value
  }

  @Metadata(virtual = false)
  class StringValue(target: String)
    extends ConstantValue[String](PropertyKind.STRING, target) {
    def this() = this(null)

    @BeanProperty
    @Property
    var value: String = _
  }

  @Metadata(virtual = false)
  class PasswordValue(target: String)
    extends ConstantValue[String](PropertyKind.STRING, target) {
    def this() = this(null)

    @BeanProperty
    @Property(password = true)
    var value: String = _

  }

  object PasswordValue {
    def apply(target: String, value: String): PasswordValue = {
      val pass = new PasswordValue(target)
      pass.value = PasswordEncrypter.getInstance().ensureEncrypted(value)
      pass
    }
  }

  @Metadata(virtual = false)
  class BooleanValue(target: String)
    extends ConstantValue[Boolean](PropertyKind.BOOLEAN, target) {
    def this() = this(null)

    @BeanProperty
    @Property
    var value: Boolean = _
  }

  @Metadata(virtual = false)
  class DateValue(target: String)
    extends ConstantValue[Date](PropertyKind.DATE, target) {
    def this() = this(null)

    @BeanProperty
    @Property
    var value: Date = _

  }

  @Metadata(virtual = false)
  class IntegerValue(target: String)
    extends ConstantValue[Integer](PropertyKind.INTEGER, target) {
    def this() = this(null)

    @BeanProperty
    @Property
    var value: Integer = _
  }

  // TODO: ENUM?

  @Metadata(virtual = false)
  class ListStringValue(target: String)
    extends ConstantValue[java.util.List[String]](PropertyKind.LIST_OF_STRING, target) {
    def this() = this(null)

    @BeanProperty
    @Property
    var value: java.util.List[String] = _

  }

  @Metadata(virtual = false)
  class SetStringValue(target: String)
    extends ConstantValue[java.util.Set[String]](PropertyKind.SET_OF_STRING, target) {
    def this() = this(null)

    @BeanProperty
    @Property
    var value: java.util.Set[String] = _

  }

  @Metadata(virtual = false)
  class MapStringStringValue(target: String)
    extends ConstantValue[java.util.Map[String, String]](PropertyKind.MAP_STRING_STRING, target) {
    def this() = this(null)

    @BeanProperty
    @Property
    var value: java.util.Map[String, String] = _

  }

  object ConstantValue {
    def apply[Value <: Any](target: String, value: Value): ConstantValue[Value] = {
      value match {
        case v: String =>
          val m = new StringValue(target)
          m.value = v
          m

        case v: Boolean =>
          val m = new BooleanValue(target)
          m.value = v
          m
        case v: java.lang.Boolean =>
          val m = new BooleanValue(target)
          m.value = v.booleanValue()
          m

        case v: Date =>
          val m = new DateValue(target)
          m.value = v
          m

        case v: Integer =>
          val m = new IntegerValue(target)
          m.value = v
          m

        case v: java.util.List[String@unchecked] =>
          val m = new ListStringValue(target)
          m.value = v
          m

        case v: java.util.Set[String@unchecked] =>
          val m = new SetStringValue(target)
          m.value = v
          m

        case v: java.util.Map[String@unchecked, String@unchecked] =>
          val m = new MapStringStringValue(target)
          m.value = v
          m
      }
    }.asInstanceOf[ConstantValue[Value]]
  }

}
