package com.xebialabs.xlrelease.webhooks.configuration.jms

import com.xebialabs.xlplatform.webhooks.events.handlers.{EventDispatcher, EventPublisher}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.webhooks.configuration.jms.WebhookJmsConfiguration.WEBHOOK_EVENT_QUEUE_NAME
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.jms.annotation.EnableJms
import org.springframework.jms.config.{DefaultJmsListenerContainerFactory, JmsListenerEndpointRegistry}
import org.springframework.jms.connection.CachingConnectionFactory
import org.springframework.jms.core.JmsTemplate
import org.springframework.jms.support.converter.MessageConverter
import org.springframework.jms.support.destination.{BeanFactoryDestinationResolver, DestinationResolver}

import jakarta.jms.{ConnectionFactory, Destination}

object WebhookJmsConfiguration {
  final val WEBHOOK_EVENT_QUEUE_NAME = "webhookEventQueue"
}

@Configuration
@ConditionalOnProperty(name = Array("xl.features.webhooks.event-publisher.type"), havingValue = "jms")
@EnableJms
class WebhookJmsConfiguration(xlrConfig: XlrConfig, eventDispatcher: EventDispatcher) {

  val config: WebhookQueueConfig = {
    WebhookQueueConfig.fromConfig(xlrConfig.xl)
  }

  lazy val delegate: BasicWebhookJmsConfiguration = {
    config match {
      case c: RabbitMqWebhookQueueConfig => new RabbitWebhookJmsConfiguration(c)
      case c: ActiveMqWebhookQueueConfig => new ActiveMqWebhookJmsConfiguration(c)
      case c: EmbeddedActiveMqWebhookQueueConfig => new EmbeddedActiveMqWebhookJmsConfiguration(c)
    }
  }

  @Bean
  def jmsConnectionFactory: ConnectionFactory = {
    delegate.jmsConnectionFactory
  }

  @Bean(name = Array(WEBHOOK_EVENT_QUEUE_NAME))
  def webhookEventQueue: Destination = delegate.webhookEventQueue

  @Bean
  lazy val converter: MessageConverter = EventMessageConverter

  @Bean
  def jmsListenerContainerFactory: DefaultJmsListenerContainerFactory = {
    val factory = new DefaultJmsListenerContainerFactory()
    factory.setConnectionFactory(jmsConnectionFactory)
    factory.setDestinationResolver(destinationResolver)
    factory.setMessageConverter(converter)
    factory.setAutoStartup(false)
    factory
  }

  @Bean
  def destinationResolver: DestinationResolver = {
    new BeanFactoryDestinationResolver()
  }

  @Bean
  def jmsTemplate: JmsTemplate = {
    val cf = if (config.caching) {
      new CachingConnectionFactory(delegate.jmsConnectionFactory)
    } else {
      delegate.jmsConnectionFactory
    }
    val template = new JmsTemplate(cf)
    template.setDestinationResolver(destinationResolver)
    template.setMessageConverter(converter)
    template
  }

  @Bean
  def eventPublisher: EventPublisher = {
    new JmsEventPublisher(jmsTemplate, eventDispatcher)
  }

  @Bean
  def jmsListenerLifecycleManager(registry: JmsListenerEndpointRegistry) : JmsListenerLifecycleManager = {
    new JmsListenerLifecycleManager(registry)
  }
}
