package com.xebialabs.xlrelease.webhooks.configuration.jms

import com.xebialabs.xlplatform.webhooks.events.domain.{Event, EventSource}
import com.xebialabs.xlplatform.webhooks.events.handlers.{EventDispatcher, EventPublisher}
import grizzled.slf4j.Logging
import org.springframework.jms.annotation.JmsListener
import org.springframework.jms.core.JmsTemplate

/**
 * This is a component that publishes events across the system to all nodes so that all interested event consumers would receive this event.
 * Once the event is published it should be picked by some component and forwarded to event dispatcher that will dispatch the event to event consumers.
 *
 * In this particular case JMS is transport mechanism that offers some level of reliability so this class is at the same time publishing event to a queue
 * and listens on the same queue so that it can dispatch event via event dispatcher.
 *
 * @param jmsTemplate JMS template
 * @param eventDispatcher Event dispatcher
 */
class JmsEventPublisher(jmsTemplate: JmsTemplate, eventDispatcher: EventDispatcher) extends EventPublisher with Logging {

  override def publishEvent(eventSource: EventSource, event: Event): Boolean = {
    logger.debug(s"publishEvent($eventSource, $event)")
    jmsTemplate.convertAndSend(WebhookJmsConfiguration.WEBHOOK_EVENT_QUEUE_NAME, event)
    true
  }

  @JmsListener(destination = WebhookJmsConfiguration.WEBHOOK_EVENT_QUEUE_NAME)
  def listen(event: Event): Unit = {
    logger.debug(s"Received event: $event")
    eventDispatcher.dispatch(event)
  }
}

