package com.xebialabs.xlrelease.webhooks.mapping

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind.{CI, LIST_OF_CI, SET_OF_CI}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xlrelease.domain.variables._
import grizzled.slf4j.Logging

import java.util
import java.util.Date
import scala.annotation.tailrec
import scala.collection.mutable
import scala.jdk.CollectionConverters._


case class PropertyAddress private(init: Seq[PropertyDef], last: PropertyDef)

object PropertyAddress extends Logging {
  def apply(a: String): PropertyAddress = {
    if (a.contains('.')) {
      if (a.contains("..")) {
        throw new IllegalArgumentException(s"Malformed property address, it contains two consecutive dots: '$a'")
      } else {
        val addr = a.split("\\.(?![^\\[\\]]*])").toSeq.map(PropertyDef.apply)
        PropertyAddress(addr.init, addr.last)
      }
    } else {
      PropertyAddress(Seq.empty[PropertyDef], PropertyDef(a))
    }
  }

  private lazy val ciKinds = Set(CI, LIST_OF_CI, SET_OF_CI)

  @tailrec
  final def hasProperty(ciType: Type, propertyAddress: PropertyAddress): Boolean = {
    val properties = ciType.getDescriptor.getPropertyDescriptors.asScala.map(pd => pd.getName -> pd).toMap
    if (propertyAddress.init.isEmpty) {
      properties.contains(propertyAddress.last.name)
    } else {
      properties.get(propertyAddress.init.head.name) match {
        case None => false
        case Some(pd) =>
          if (ciKinds contains pd.getKind) {
            hasProperty(pd.getReferencedType, propertyAddress.copy(init = propertyAddress.init.tail))
          } else {
            false
          }
      }
    }
  }

  def getProperty(sourceProperty: String): BaseConfigurationItem => AnyRef =
    getCiProperty(sourceProperty)


  def setProperty(targetProperty: String): BaseConfigurationItem => Any => Unit =
    setCiProperty(targetProperty)

  private def getCiProperty(sourceProperty: String): BaseConfigurationItem => AnyRef = {
    val addr = PropertyAddress(sourceProperty)
    getCiValue(addr.init) andThen
      getValue(addr.last)
  }

  private def setCiProperty(targetProperty: String): BaseConfigurationItem => Any => Unit = {
    val addr = PropertyAddress(targetProperty)
    getCiValue(addr.init) andThen
      setValue(addr.last)
  }

  def any2AnyRef(value: Any): AnyRef = value match {
    case bool: Boolean => java.lang.Boolean.valueOf(bool)
    case ref: AnyRef => ref
    case null => null
  }

  def any2Any(value: Any): Any = value match {
    case bool: java.lang.Boolean => bool.booleanValue()
    case ref: AnyRef => ref
    case null => null
  }

  def isListOfVariables(propertyDef: PropertyDef): BaseConfigurationItem => Boolean = propertyDef match {
    case PropertyDef("id", _) =>
      _ => throw new IllegalArgumentException(s"isListOfVariables is not applicable to id property.")
    case other =>
      ci => {
        val propertyDescriptor = ci.getType.getDescriptor.getPropertyDescriptor(other.name)
        val referencedType = propertyDescriptor.getReferencedType
        propertyDescriptor.getKind.equals(LIST_OF_CI) &&
          referencedType != null &&
          referencedType.equals(Type.valueOf(classOf[Variable]))
      }
  }

  def getSingleProperty(propertyDef: PropertyDef): BaseConfigurationItem => Any = propertyDef match {
    case PropertyDef("id", Some(key)) =>
      _ => throw new IllegalArgumentException(s"Cannot access element '$key' of 'id'.")
    case PropertyDef("id", None) =>
      ci => {
        val id = ci.getId
        logger.trace(s"getSingleProperty fetched id of '${ci.getType}#${ci.getId}'")
        id
      }
    case PropertyDef(name, _) =>
      ci => {
        val value = ci.getProperty[Any](name)
        logger.trace(s"getSingleProperty fetched property '$name': of '${ci.getType}#${ci.getId}'")
        value
      }
  }

  def setSingleProperty(propertyDef: PropertyDef): BaseConfigurationItem => Any => Unit = propertyDef match {
    case PropertyDef("id", Some(key)) =>
      _ => _ => throw new IllegalArgumentException(s"Cannot access element '$key' of 'id'")
    case PropertyDef("id", None) =>
      ci =>
        value => {
          logger.trace(s"setSingleProperty setting id '${value}' to '${ci.getType}#${ci.getId}'")
          ci.setId(value.toString)
        }
    case PropertyDef(name, _) =>
      ci =>
        value => {
          logger.trace(s"setSingleProperty setting property '$name' to '${ci.getType}#${ci.getId}'")
          ci.setProperty(name, value)
        }
  }

  private def getValue(propertyDef: PropertyDef): BaseConfigurationItem => AnyRef = {
    ci =>
      logger.trace("getValue '$propertyDef' of '${ci.getType}#${ci.getId}'")
      val value: AnyRef = cloneValue(any2AnyRef(getSingleProperty(propertyDef).apply(ci)))

      propertyDef.key.fold(value) { key =>
        any2AnyRef {
          value match {
            case map: java.util.Map[String@unchecked, Any@unchecked] =>
              map.get(key)

            case list: java.util.List[Any@unchecked] =>
              val index = Integer.parseInt(key)
              list.get(index)

            case set: java.util.Set[Any@unchecked] =>
              val scalaSet = set.asScala
              scalaSet.collectFirst {
                case element if element.toString == key =>
                  element
              }.getOrElse {
                throw new NotFoundException(s"Element '$key' not found in ${scalaSet.mkString("{", ", ", "}")}")
              }

            case other =>
              if (other != null) {
                throw new IllegalArgumentException(
                  s"Property '${propertyDef.name}' of '${ci.getType}#${ci.getId}' is not a collection (${other.getClass.getSimpleName})." +
                    s" Cannot access element with key '$key'"
                )
              } else {
                throw new IllegalArgumentException(
                  s"Property '${propertyDef.name}' of '${ci.getType}#${ci.getId}' is null." +
                    s" Cannot access element with key '$key'"
                )
              }
          }
        }
      }
  }

  private def setValue(propertyDef: PropertyDef): BaseConfigurationItem => Any => Unit = {
    ci =>
      val property: AnyRef = any2AnyRef(getSingleProperty(propertyDef).apply(ci))
      value =>
        logger.trace("setValue '$propertyDef' of '${ci.getType}#${ci.getId}'")
        val anyValue = any2Any(value)
        val mutatedValue = propertyDef.key.fold(anyValue) { key =>
          property match {
            case map: java.util.Map[String@unchecked, Any@unchecked] =>
              map.put(key, anyValue)
              map

            case list: java.util.List[Any@unchecked] =>
              // there is specific case when this list is list of xlrelease.Variable
              if (isListOfVariables(propertyDef).apply(ci)) {
                val listOfVariables: mutable.Seq[Variable] = list.asInstanceOf[java.util.List[Variable@unchecked]].asScala
                val variable: Variable = listOfVariables
                  .find(_.getKey.equals(key))
                  .getOrElse {
                    val variable = makeVariableForValue(key, value)
                    list.add(variable)
                    variable
                  }
                variable.setUntypedValue(value)
              } else {
                if (key.isEmpty) {
                  list.add(anyValue)
                } else {
                  val index = Integer.parseInt(key)
                  list.set(index, anyValue)
                }
              }
              list

            case set: java.util.Set[Any@unchecked] =>
              val scalaSet = set.asScala
              if (key.isEmpty) {
                (set.asScala + anyValue).asJava
              } else {
                if (scalaSet.exists(_.toString == key)) {
                  (set.asScala.filterNot(_.toString == key) + anyValue).asJava
                } else {
                  throw new NotFoundException(s"Element '$key' not found in ${scalaSet.mkString("{", ", ", "}")}")
                }
              }

            case other =>
              throw new IllegalArgumentException(
                s"Property '${propertyDef.name}' of '${ci.getType}#${ci.getId}' is not a collection, but it is ${getClassSimpleName(other)}." +
                  s" Cannot access element with key '$key'"
              )
          }
        }
        setSingleProperty(propertyDef).apply(ci).apply(mutatedValue)
  }

  def makeVariableForValue(key: String, value: Any): Variable = {
    var variable = value match {
      case _: String => new StringVariable()
      case _: Boolean => new BooleanVariable()
      case _: Date => new DateVariable()
      case _: Integer => new IntegerVariable()
      case _: java.util.List[_] => new ListStringVariable()
      case _: java.util.Map[_, _] => new MapStringStringVariable()
      case _: java.util.Set[_] => new SetStringVariable()
      case _ => new StringVariable()
    }
    variable.setKey(key)
    variable
  }

  // traverse the address by using the previous value as the ConfigurationItem for the next iteration.
  def getCiValue(address: Seq[PropertyDef]): BaseConfigurationItem => BaseConfigurationItem = {
    ci =>
      address
        .zipWithIndex
        .foldLeft(ci) {
          case (ci1, (propertyDef, i)) =>
            getValue(propertyDef).apply(ci1) match {
              case ci2: BaseConfigurationItem =>
                logger.trace(s"getCiValue picking '${propertyDef}' from '${ci1.getType}#${ci1.getId}', got '${ci2.getType}#${ci2.getId}'")
                ci2
              case other =>
                throw new IllegalArgumentException(
                  s"Property '$propertyDef' of '${ci1.getType}#${ci.getId}' is not a CI but it is ${getClassSimpleName(other)}." +
                    s" Cannot access nested property in '${address.splitAt(i)._2.tail.mkString(".")}'"
                )
            }
        }
  }

  def cloneValue(value: AnyRef): AnyRef = value match {
    case map: java.util.Map[String@unchecked, Any@unchecked] =>
      val newMap = new java.util.HashMap[String, Any]()
      newMap.putAll(map)
      newMap

    case list: java.util.List[Any@unchecked] =>
      val newList = new java.util.ArrayList[Any]()
      newList.addAll(list)
      newList

    case set: java.util.Set[Any@unchecked] =>
      val newSet = new util.HashSet[Any]()
      newSet.addAll(set)
      newSet

    case other =>
      other
  }

  private def getClassSimpleName(obj: AnyRef): String = Option(obj).map(o => s"(${o.getClass.getSimpleName})").getOrElse("null")

}
