package com.xebialabs.xlrelease.webhooks.configuration.jms

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.xlrelease.support.config.TypesafeConfigExt._
import org.slf4j.{Logger, LoggerFactory}

import java.net.URI

sealed trait WebhookQueueConfig {

  def conf: Config

  def caching: Boolean = conf.getOptionalBoolean("caching").getOrElse(false)

  def url: String = conf.getString("url")

  def username: String = conf.getString("username")

  def password: String = conf.getString("password")

  def queueName: String = conf.getString("queueName")

  def useSsl: Boolean = {
    val brokerUri = new URI(url)
    val scheme = brokerUri.getScheme
    // ssl configuration can be enabled explicitly, but cannot be disabled if specified via URL
    scheme.contains("amqps") || scheme.contains("ssl") || sslConfig.getOptionalBoolean("enabled").getOrElse(false)
  }

  def sslConfig: Config = conf.getOptionalConfig("ssl").getOrElse(ConfigFactory.empty)

}

case class RabbitMqWebhookQueueConfig(conf: Config) extends WebhookQueueConfig {

  def queueType: String = conf.getOptionalString("queueType").getOrElse("classic")

}

case class EmbeddedActiveMqWebhookQueueConfig(conf: Config) extends WebhookQueueConfig

case class ActiveMqWebhookQueueConfig(conf: Config) extends WebhookQueueConfig

object WebhookQueueConfig {
  protected val logger: Logger = LoggerFactory.getLogger(this.getClass)

  def fromConfig(config: Config): WebhookQueueConfig = {
    try {
      val conf = config.getConfig("queue")
      if (conf.getBoolean("embedded")) {
        EmbeddedActiveMqWebhookQueueConfig(conf)
      } else {
        conf.getOptionalString("connector") match {
          case Some("activemq-jms") => ActiveMqWebhookQueueConfig(conf)
          case Some("rabbitmq-jms") => RabbitMqWebhookQueueConfig(conf)
          case None =>
            logger.warn("'xl.queue.connector' was not specified defaulting to 'rabbitmq-jms'")
            RabbitMqWebhookQueueConfig(conf)
          case _ =>
            val errorMsg = "Connector was not specified. Please set `xl.queue.connector` to one of the following values: 'rabbitmq-jms', 'activemq-jms'"
            throw new IllegalStateException(errorMsg)
        }
      }
    } catch {
      case e: Throwable =>
        logger.warn("Unable to parse 'xl.queue' configuration", e)
        throw e
    }
  }
}
