package com.xebialabs.xlrelease.webhooks.utils

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.platform.script.jython.Syntactic.wrapperCodeWithLib
import com.xebialabs.platform.script.jython.{ConsumerWriter, ScriptSource}
import com.xebialabs.xlplatform.webhooks.domain.Endpoint
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.script.builder.ScriptContextBuilder
import com.xebialabs.xlrelease.script.jython.{JythonScriptExecutor, JythonScriptHelper}
import com.xebialabs.xlrelease.script.{XlrScript, XlrScriptContext}
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xlrelease.webhooks.authentication.BaseRequestAuthentication
import com.xebialabs.xlrelease.webhooks.consumers.BaseProcessor
import com.xebialabs.xlrelease.webhooks.utils.JythonScriptEventProcessingUtils.ciHelper
import grizzled.slf4j.Logger
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.{lang, util}
import javax.script.{ScriptContext, ScriptException}
import scala.jdk.CollectionConverters._
import scala.language.postfixOps

object JythonScriptEventProcessingUtils {
  lazy val ciHelper: ScriptSource = ScriptSource.byResource("ciHelper.py")
}

@Component
class JythonScriptEventProcessingUtils @Autowired()(jythonScriptExecutor: JythonScriptExecutor) {

  def execute[A](xlrScriptContext: XlrScriptContext)
                (outputProcessor: (String, PartialFunction[AnyRef, A])): Option[A] = {
    val (output, processor) = outputProcessor
    xlrScriptContext.addScript(XlrScript("<event_output_unwrap>", JythonScriptHelper.unwrapBindings(output :: Nil), checkPermissions = false, wrap = false))
    jythonScriptExecutor.evalScript(xlrScriptContext)
    Option(resultProcessor[A](output)(processor).apply(xlrScriptContext)).flatten
  }

  def executeFilter(config: BaseProcessor, event: Event, sandbox: Boolean): Logger => ScriptSource => Boolean =
    logger => {
      scriptSource => {
        try {
          val scriptContext = processorContext(config, event).apply(logger)
          scriptContext.addScript(XlrScript("<event_filter>", scriptSource, sandbox, wrap = true))
          execute[Boolean](scriptContext)("accepted" -> booleanResult).getOrElse(false)
        } catch {
          case ex: ScriptException =>
            logger.warn(ex.getMessage, ex)
            false
        }
      }
    }

  def executeMap(config: BaseProcessor, event: Event, sandbox: Boolean): Logger => ScriptSource => Event =
    logger => {
      scriptSource => {
        try {
          val scriptContext = processorContext(config, event, libraries = ciHelper :: Nil).apply(logger)
          scriptContext.addScript(XlrScript("<event_mapping>", scriptSource, sandbox, wrap = true))
          execute[Event](scriptContext)(
            "output" -> {
              case output: Event if output.getType.instanceOf(Type.valueOf(config.outputEventType)) =>
                CiHelper.fixUpInternalReferences(output)
                output
            }
          ).orNull
        } catch {
          case ex: ScriptException =>
            logger.warn(ex.getMessage, ex)
            null
        }
      }
    }

  def executeAuthentication(config: BaseRequestAuthentication,
                            endpoint: Endpoint,
                            headers: util.Map[String, String],
                            params: util.Map[String, Array[String]],
                            payload: String,
                            sandbox: Boolean): Logger => ScriptSource => Boolean =
    logger => {
      scriptSource =>
        try {
          val scriptContext: XlrScriptContext = createJythonContext(
            variables = Map(
              "endpoint" -> endpoint,
              "config" -> config,
              "headers" -> headers,
              "parameters" -> Option(params).map(_.asScala.view.mapValues(_.mkString(", ")).toMap.asJava).orNull,
              "payload" -> payload
            )
          ).apply(logger)
          scriptContext.addScript(XlrScript("<event_authentication>", scriptSource, sandbox, wrap = true))
          execute[Boolean](scriptContext)("authenticated" -> booleanResult).getOrElse(false)
        } catch {
          case ex: ScriptException =>
            logger.warn(ex.getMessage, ex)
            false
        }
    }

  def scriptLocationByConvention(pluginType: Type): String =
    pluginType.getPrefix + Ids.SEPARATOR + pluginType.getName + ".py"

  def nonEmptyScriptLocation(location: String): Option[String] =
    Option(location)
      .filter(_.nonEmpty)

  def scriptLocationOrDefault(pluginType: Type, location: String): String =
    nonEmptyScriptLocation(location).getOrElse(scriptLocationByConvention(pluginType))

  def resultProcessor[A](name: String)(f: PartialFunction[AnyRef, A]): ScriptContext => Option[A] =
    ctx => f.lift(ctx.getAttribute(name))

  def booleanResult: PartialFunction[AnyRef, Boolean] = {
    case bool: lang.Boolean =>
      bool
  }

  def processorContext(config: BaseProcessor, event: Event, libraries: Seq[ScriptSource] = Nil): Logger => XlrScriptContext =
    createJythonContext(
      variables = Map(
        "config" -> config,
        "input" -> event
      ),
      libraries = libraries
    )

  def createJythonContext(variables: Map[String, Any], libraries: Seq[ScriptSource] = Seq.empty): Logger => XlrScriptContext = logger => {
    EventScriptContext.eventContext(logger, variables, libraries)
  }
}

object EventScriptContext {

  def eventContext(logger: Logger, variables: Map[String, Any], libraries: Seq[ScriptSource] = Seq.empty): XlrScriptContext = {
    new EventScriptContextBuilder(logger, variables, libraries).build()
  }

  private class EventScriptContextBuilder(logger: Logger, variables: Map[String, Any], libraries: Seq[ScriptSource] = Seq.empty) extends ScriptContextBuilder {
    withLogger()
    withPythonSugar().withPythonGlobals()

    override protected def doBuild(scriptContext: XlrScriptContext): Unit = {
      variables.foreach {
        case (name, value) => scriptContext.setAttribute(name, value, ScriptContext.ENGINE_SCOPE)
      }
      scriptContext.setWriter(new ConsumerWriter(text => {
        val stripped = text.stripSuffix("\n")
        if (!stripped.forall(Set(' ', '\n') contains)) {
          logger.info(stripped)
        }
      }))
      scriptContext.setErrorWriter(new ConsumerWriter(text => logger.error(text.stripSuffix("\n"))))
      (libraries ++ wrapperCodeWithLib(variables.keys)).foreach { lib =>
        val script = XlrScript("", lib, checkPermissions = false, wrap = false)
        scriptContext.addScript(script)
      }
    }

  }
}
