package com.xebialabs.xlrelease.webhooks.repository.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlplatform.utils.ResourceManagement
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.webhooks.repository.persistence.WebhookSchema.{CI_EVENT_RECORDS => CER}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import java.util.Date

case class CiEventRecordRow(eventId: CiId, sourceUid: CiUid, eventTime: Date, content: String) {
  def contentToEvent: Event = {
    Event.decode(content)
  }
}

// Records jms messages listened to by CiEventConsumers
@Component
@Transactional
class CiEventRecordsPersistence @Autowired()(configurationPersistence: ConfigurationPersistence,
                                             @Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                             @Qualifier("xlrRepositorySqlDialect") val dialect: Dialect)
  extends PersistenceSupport
    with LimitOffset {

  def insert(event: Event): Unit = {
    configurationPersistence.getUid(event.getSourceId) match {
      case None => throw new NotFoundException(s"Source [${event.getSourceId}] not found")
      case Some(sourceUid) =>
        sqlExecWithContent(
          s"""INSERT INTO ${CER.TABLE}
             | (
             | ${CER.EVENT_ID}
             | , ${CER.CONFIGURATION_UID}
             | , ${CER.EVENT_TIME}
             | , ${CER.CONTENT}
             | ) VALUES (
             | :${CER.EVENT_ID}
             | , :${CER.CONFIGURATION_UID}
             | , :${CER.EVENT_TIME}
             | , :${CER.CONTENT}
             | )""".stripMargin,
          params(
            CER.EVENT_ID -> event.getId,
            CER.CONFIGURATION_UID -> sourceUid,
            CER.EVENT_TIME -> new Date()
          ),
          CER.CONTENT -> Event.encode(event),
          _ => ()
        )
    }
  }

  def findByEventId(eventId: CiId): Option[CiEventRecordRow] = {
    sqlQuery(
      s"""SELECT ${CER.EVENT_ID}
         | , ${CER.CONFIGURATION_UID}
         | , ${CER.EVENT_TIME}
         | , ${CER.CONTENT}
         | FROM ${CER.TABLE}
         | WHERE ${CER.EVENT_ID} = :${CER.EVENT_ID}
         |""".stripMargin,
      params(CER.EVENT_ID -> eventId),
      webhookEventRecordRowMapper
    ).headOption
  }

  private def generateDeleteOldestRecordsQuery(recordsToKeep: Int) = {
    s"""DELETE FROM ${CER.TABLE}
       | WHERE ${CER.CONFIGURATION_UID} = :${CER.CONFIGURATION_UID} AND
       | ${CER.EVENT_ID} NOT IN (
       |  SELECT ${CER.EVENT_ID} FROM (
       |   ${
      addLimitAndOffset(
        s"""SELECT ${CER.EVENT_ID}
           |   FROM ${CER.TABLE}
           |   WHERE ${CER.CONFIGURATION_UID} = :${CER.CONFIGURATION_UID}
           |   ORDER BY ${CER.EVENT_TIME} DESC""".stripMargin,
        Some(recordsToKeep)
      )
    }
       |  ) EVENT_IDS
       | )""".stripMargin
  }

  def deleteRecordsByUuid(uid: CiUid, recordsToKeep: Int): Unit = {
    val query = generateDeleteOldestRecordsQuery(recordsToKeep)
    sqlExec(query, params(CER.CONFIGURATION_UID -> uid), _.execute())
  }

  private val webhookEventRecordRowMapper: RowMapper[CiEventRecordRow] = (rs: ResultSet, _: Int) => {
    CiEventRecordRow(
      eventId = rs.getString(CER.EVENT_ID),
      sourceUid = rs.getInt(CER.CONFIGURATION_UID),
      eventTime = rs.getDate(CER.EVENT_TIME),
      content = ResourceManagement.using(rs.getBinaryStream(CER.CONTENT))(decompress)
    )
  }
}
