package com.xebialabs.xlplatform.webhooks.endpoint

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlplatform.webhooks.endpoint.exceptions.{EndpointNotFound, UnauthorizedRequest}
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.webhooks.consumers.logging.{WebHookRequestIgnoredEvent, WebHookRequestUnauthorizedEvent}
import grizzled.slf4j.Logging

import jakarta.servlet.http.HttpServletRequest
import jakarta.ws.rs.core.Response
import scala.util.Try

trait RequestAcceptor extends Logging {

  val eventBus: EventBus

  def accept(path: String, request: HttpServletRequest): Response = {
    processRequest(path, request).recover {
      case e: exceptions.WebhookEndpointControllerException =>
        logger.warn(e.getMessage, if (e.logStackTrace) e else null)
        Response.status(e.status).entity(e.getMessage).build()

      case e: NotFoundException =>
        logger.warn(e.getMessage)
        Response.status(Response.Status.NOT_FOUND).entity(e.getMessage).build()

      case e: Throwable =>
        logger.warn(e.getMessage, e)
        Response.status(Response.Status.INTERNAL_SERVER_ERROR)
          .entity("Exception happened when trying to process your request, check XLRelease log for details").build()
    }.get
  }

  def requestPrefix(request: HttpServletRequest) = s"${request.getRequestURI} from ${request.getRemoteAddr}:${request.getRemotePort}"

  protected def processRequest(str: String, request: HttpServletRequest): Try[Response]

  protected def processError(ex: Throwable): Unit =
    ex match {
      case ex: EndpointNotFound =>
        eventBus.publish(WebHookRequestIgnoredEvent(ex.getMessage))
      case ex: UnauthorizedRequest =>
        eventBus.publish(WebHookRequestUnauthorizedEvent(ex.getMessage))
      case _ =>
        eventBus.publish(WebHookRequestIgnoredEvent(ex.getMessage))
    }
}
