package com.xebialabs.xlrelease.webhooks.endpoint

import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.webhooks.authentication.RequestAuthenticationMethod
import com.xebialabs.xlplatform.webhooks.domain.{Endpoint, HttpRequestEvent}
import com.xebialabs.xlplatform.webhooks.endpoint.WebhooksEndpointController
import com.xebialabs.xlplatform.webhooks.endpoint.exceptions.EndpointDisabled
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.service.ConfigurationVariableService
import com.xebialabs.xlrelease.webhooks.consumers.logging.WebHookRequestAcceptedEvent
import com.xebialabs.xlrelease.webhooks.jms.JmsEventPublisherHandler
import com.xebialabs.xlrelease.webhooks.registry.EndpointRegistry
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.jms.core.JmsTemplate
import org.springframework.stereotype.Controller

import jakarta.servlet.http.HttpServletRequest
import jakarta.ws.rs._
import jakarta.ws.rs.core.{MediaType, Response}
import scala.util.{Failure, Success, Try}

@Path("/webhooks")
@Produces(Array(MediaType.APPLICATION_JSON))
@Consumes(Array(MediaType.APPLICATION_JSON))
@Controller
class WebhooksResource @Autowired()(endpointRegistry: EndpointRegistry,
                                    configurationVariableService: ConfigurationVariableService,
                                    override val eventBus: XLReleaseEventBus,
                                    val jmsTemplate: JmsTemplate,
                                    // we need to autowire this here to make sure, that Jython
                                    // scripting components are ready by the time we receive
                                    // first call
                                    val authenticationMethods: java.util.List[RequestAuthenticationMethod])
  extends WebhooksEndpointController(endpointRegistry)
    with JmsEventPublisherHandler[HttpRequestEvent, Endpoint] {

  override protected def processRequest(path: String, request: HttpServletRequest): Try[Response] = {
    val result = super.processRequest(path, request)
    result match {
      case Success(_) =>
        eventBus.publish(WebHookRequestAcceptedEvent(s"Accepted WebHook request for path: webhooks/$path"))
      case Failure(ex) =>
        processError(ex)
    }
    result
  }

  override protected def getEndpoint(path: String): Try[Endpoint] = {
    val triedEndpoint = super.getEndpoint(path)
    triedEndpoint.foreach {
      case e: WebhookEndpoint => configurationVariableService.resolve(e)
      case _ => ()
    }
    triedEndpoint
  }

  override protected def checkEnabled(endpoint: Endpoint): Try[Unit] =
    if (NodeState.isActive) {
      super.checkEnabled(endpoint)
    } else {
      Failure(EndpointDisabled(endpoint))
    }
}
