package com.xebialabs.xlrelease.webhooks.untyped

import com.jayway.jsonpath.JsonPath
import com.jayway.jsonpath.spi.json.JsonProvider
import com.xebialabs.xlplatform.webhooks.domain.HttpRequestEvent
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Service
class EventContentParser @Autowired()(val jsonProvider: JsonProvider) extends Logging {
  def parse(event: Event): EventDocument = {
    event match {
      case httpEvent: HttpRequestEvent =>
        Map[String, Any](
          "event" -> parseContent(httpEvent),
          "payload" -> httpEvent.content,
          "headers" -> httpEvent.headers,
          "parameters" -> httpEvent.parameters
        ).asJava
      case _ =>
        // TODO: fully implement later
        Map[String, Any]("event" -> event).asJava
    }
  }

  private def parseContent(httpEvent: HttpRequestEvent): AnyRef = {
    // currently we only support JSON as content type, add later dynamic conversion if needed
    val content = if (hasText(httpEvent.content)) httpEvent.content else EMPTY_JSON_CONTENT
    Try(jsonProvider.parse(content)) match {
      case Success(value) => value
      case Failure(exception) =>
        logger.debug(s"Error parsing event body as JSON. Error: ${exception.getMessage}. Body: $content")
        Map.empty[String, Any].asJava
    }
  }
}

trait PropertyAccessor {
  def getProperty(path: String): Try[Any]
}

class JsonPathEventAccessor(document: EventDocument) extends PropertyAccessor {

  override def getProperty(path: String): Try[Any] = {
    val modifiedPath = path.replace(".^", "")
    Try(JsonPath.read[Any](document, s"$$.$modifiedPath"))
  }

}


