package com.xebialabs.xlrelease.webhooks.mapping

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xlplatform.webhooks.domain.HttpRequestEvent
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.repository.Ids.findFolderId
import com.xebialabs.xlrelease.script.ScriptVariables
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.variable.VariableHelper._
import com.xebialabs.xlrelease.webhooks.mapping.MappedProperty.{ConstantValue, PropertyValue, VariableValue}
import com.xebialabs.xlrelease.webhooks.mapping.PropertyAddress.{any2AnyRef, getProperty, setProperty}
import com.xebialabs.xlrelease.webhooks.mapping.VariableResolution.freezeVariables
import com.xebialabs.xlrelease.webhooks.untyped.{EventContentParser, JsonPathEventAccessor, PropertyAccessor}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils.hasText

import java.util
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
class PropertiesMapper @Autowired()(implicit val scriptVariables: ScriptVariables,
                                    implicit val eventParser: EventContentParser) extends Logging {

  def mapProperties(config: BaseConfigurationItem, event: Event, target: BaseConfigurationItem): Unit = {
    require(config.hasProperty("mappedProperties"))

    val mappedProperties = config.getProperty[util.List[MappedProperty]]("mappedProperties").asScala

    val context = new EventVariableResolver(event, findFolderId(config.getId))

    lazy val _getConstantValue: ConstantValue[Any] => Try[Any] = getConstantValue(context)
    lazy val _getVariableValue: VariableValue => Try[Any] = getVariableValue(context)
    lazy val _getPropertyValue: PropertyValue => Try[Any] = getPropertyValue(context)

    mappedProperties.foreach { mappedProperty =>
      mappedProperty.fold[Try[Any]](_getConstantValue, _getVariableValue, _getPropertyValue)
        .map { value =>
          logger.debug(s"setting '$value' on '$target' using $mappedProperty")
          setProperty(mappedProperty.targetProperty).apply(target).apply(value)
        }.get
    }
  }

  protected def getConstantValue(implicit context: EventVariableResolver): ConstantValue[Any] => Try[Any] =
    mappedProperty => Success(freezeVariables(mappedProperty.value, failIfUnresolved = true))

  protected def getVariableValue(implicit context: EventVariableResolver): VariableValue => Try[Any] =
    variableValue => context.resolve(variableValue.variableKey)

  protected def getPropertyValue(implicit context: EventVariableResolver): PropertyValue => Try[Any] =
    mappedProperty =>
      Try(getProperty(mappedProperty.sourceProperty).apply(context.event))

}

object VariableResolution extends Logging {
  def freezeVariables[T](value: T, failIfUnresolved: Boolean = false)(implicit variableResolver: VariableResolver): T = {
    val variables = collectVariables(any2AnyRef(value))

    val resolvedVariables = variables.asScala
      .flatMap { key =>
        variableResolver.resolve(withoutVariableSyntax(key))
          .map(value => key -> VariableHelper.toString(value)).toOption
      }.toMap

    val unresolvedVariables = new util.HashSet[String]()

    val replacedValue = replaceAll(value, resolvedVariables.asJava, unresolvedVariables, freezeEvenIfUnresolved = false)

    val failedResolutions = unresolvedVariables.asScala.filter(key => variableResolver.canResolve(withoutVariableSyntax(key)))
    if (failedResolutions.nonEmpty) {
      val msg = s"Unable to resolve variables ${failedResolutions.asJava} in value '$value'."
      logger.warn(msg)
      if (failIfUnresolved) {
        throw new IllegalArgumentException(msg)
      }
    }
    replacedValue
  }
}

trait VariableResolution {
  @BeanProperty
  @Autowired
  @transient
  implicit var scriptVariables: ScriptVariables = _
  @transient
  lazy implicit val variableResolver: VariableResolver = new VariableResolver(folderId)

  def folderId: String

  def freezeVariables[T](value: T, failIfUnresolved: Boolean = false): T = {
    VariableResolution.freezeVariables(value, failIfUnresolved)
  }
}

class EventVariableResolver(val event: Event, folderId: String)(implicit val scriptVariables: ScriptVariables,
                                                                implicit val eventParser: EventContentParser)
  extends VariableResolver(folderId) {
  lazy val untypedEvent: PropertyAccessor = new JsonPathEventAccessor(eventParser.parse(event))

  val isUntypedEvent: Boolean = event.isInstanceOf[HttpRequestEvent]

  override def canResolve(variableKey: String): Boolean = {
    withoutVariableSyntax(variableKey).startsWith("event.") || super.canResolve(variableKey)
  }

  override def resolve(variableKey: String): Try[Any] = if (variableKey.startsWith("event.") && isUntypedEvent) {
    untypedEvent.getProperty(variableKey)
  } else {
    super.resolve(variableKey)
  }

}

class VariableResolver(folderId: String = null)(implicit scriptVariables: ScriptVariables) {
  lazy val globalVars: util.Map[String, Variable] = scriptVariables.initialGlobalVariables()
  lazy val folderVars: util.Map[String, Variable] = if (hasText(folderId)) {
    scriptVariables.initialFolderVariables(folderId)
  } else {
    Map.empty[String, Variable].asJava
  }

  def canResolve(variableKey: String): Boolean = isGlobalOrFolderVariable(variableKey)

  def resolve(variableKey: String): Try[Any] =
    if (variableKey.startsWith("global.")) {
      if (globalVars.containsKey(variableKey)) {
        Success(globalVars.get(variableKey).getValue)
      } else {
        Failure(new NotFoundException(s"Global variable $variableKey not found"))
      }
    } else if (variableKey.startsWith("folder.")) {
      if (folderVars.containsKey(variableKey)) {
        Success(folderVars.get(variableKey).getValue)
      } else {
        Failure(new NotFoundException(s"Folder variable $variableKey not found on immediate parent"))
      }
    } else {
      Failure(new IllegalArgumentException(s"Cannot handle variable with key '$variableKey'"))
    }
}

