package com.xebialabs.xlrelease.webhooks.endpoint

import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.webhooks.authentication.RequestAuthenticationMethod
import com.xebialabs.xlplatform.webhooks.domain.{Endpoint, HttpRequestEvent}
import com.xebialabs.xlplatform.webhooks.endpoint.WebhooksEndpointController
import com.xebialabs.xlplatform.webhooks.endpoint.exceptions.{EndpointDisabled, EndpointNotFound, UnauthorizedRequest}
import com.xebialabs.xlplatform.webhooks.queue.JmsEventPublisherHandler
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.service.ConfigurationVariableService
import com.xebialabs.xlrelease.webhooks.consumers.logging.{WebHookRequestAcceptedEvent, WebHookRequestIgnoredEvent, WebHookRequestUnauthorizedEvent}
import com.xebialabs.xlrelease.webhooks.registry.EndpointRegistry
import org.springframework.beans.factory.annotation.{Autowired, Value}
import org.springframework.jms.core.JmsTemplate
import org.springframework.stereotype.Controller

import javax.servlet.http.HttpServletRequest
import javax.ws.rs._
import javax.ws.rs.core.{MediaType, Response}
import scala.util.{Failure, Success, Try}

@Path("/webhooks")
@Produces(Array(MediaType.APPLICATION_JSON))
@Consumes(Array(MediaType.APPLICATION_JSON))
@Controller
class WebhooksResource @Autowired()(endpointRegistry: EndpointRegistry,
                                    configurationVariableService: ConfigurationVariableService,
                                    val eventBus: XLReleaseEventBus,
                                    val jmsTemplate: JmsTemplate,
                                    @Value("${xl.queue.queueName}") val queueName: String,
                                    // we need to autowire this here to make sure, that Jython
                                    // scripting components are ready by the time we receive
                                    // first call
                                    val authenticationMethods: java.util.List[RequestAuthenticationMethod])
  extends WebhooksEndpointController(endpointRegistry)
    with JmsEventPublisherHandler[HttpRequestEvent, Endpoint] {

  override protected def processRequest(path: String, request: HttpServletRequest): Try[Response] = {
    val result = super.processRequest(path, request)
    result match {
      case Success(_) =>
        eventBus.publish(WebHookRequestAcceptedEvent(s"Accepted WebHook request for path: webhooks/$path"))
      case Failure(ex) =>
        ex match {
          case ex: EndpointNotFound =>
            eventBus.publish(WebHookRequestIgnoredEvent(ex.getMessage))
          case ex: UnauthorizedRequest =>
            eventBus.publish(WebHookRequestUnauthorizedEvent(ex.getMessage))
          case _ =>
            eventBus.publish(WebHookRequestIgnoredEvent(ex.getMessage))
        }
    }
    result
  }

  override protected def getEndpoint(request: HttpServletRequest, path: String): Try[Endpoint] = {
    val triedEndpoint = super.getEndpoint(request, path)
    triedEndpoint.foreach {
      case e: WebhookEndpoint => configurationVariableService.resolve(e)
      case _ => ()
    }
    triedEndpoint
  }

  override protected def checkEnabled(request: HttpServletRequest, endpoint: Endpoint): Try[Unit] =
    if (NodeState.isActive) {
      super.checkEnabled(request, endpoint)
    } else {
      Failure(EndpointDisabled(request, endpoint))
    }
}
