package com.xebialabs.xlrelease.webhooks.endpoint

import com.xebialabs.deployit.plugin.api.validation.{ExtendedValidationContext, ValidationContext, Validator}
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.webhooks.endpoint.WebhookEndpointValidator.{invalidCharacter, isPathInvalid}
import com.xebialabs.xlrelease.webhooks.registry.EndpointRegistry
import grizzled.slf4j.Logging

class WebhookEndpointValidator extends Validator[WebhookEndpoint] with Logging {
  override def validate(ci: WebhookEndpoint, context: ValidationContext): Unit = {
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]
    val path = ci.getPath
    val errorMessage: Option[String] = {
      if (isPathInvalid(path)) {
        Some(
          s"""|HTTP endpoint for Webhooks's path contains invalid characters.
              |Invalid characters: ${invalidCharacter.mkString(" ")}""".stripMargin)
      } else {
        EndpointRegistry.findEndpoint(path) match {
          case Some(endpoint) if !endpoint.equals(ci) =>
            logger.debug(s"validate(${ci.getId}): Found existing HTTP endpoint for Webhooks with path '$path': ${endpoint.sourceId}'")
            endpoint match {
              case baseConfiguration: BaseConfiguration =>
                Some(s"HTTP endpoint for Webhooks path must be unique. Found same path for '${baseConfiguration.getTitle}'")
              case _ =>
                Some(s"HTTP endpoint for Webhooks path must be unique. Found same path for CI ${endpoint}")
            }
          case _ =>
            None
        }
      }
    }
    errorMessage.foreach(extendedContext.error(ci, "path", _))
  }
}

object WebhookEndpointValidator {
  lazy val invalidCharacter: Set[Char] = Set(
    '{', '}', '|', '\\', '^', '[', ']', '`',
    ';', '/', '?', ':', '@', '&', '=', '+', '$', ','
  )

  def isPathInvalid(path: String): Boolean = {
    path.exists(invalidCharacter.contains)
  }
}