package com.xebialabs.xlrelease.udm.reporting.repository.sql.queries

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasesSqlBuilder.alias
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.RELEASES
import com.xebialabs.xlrelease.repository.sql.persistence.{ReleasePersistence, ReleasesSqlBuilder}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Component

import java.sql.ResultSet
import java.util.{ArrayList => JArrayList, List => JList, Map => JMap}
import scala.jdk.CollectionConverters._

@Component
@IsTransactional
class LiveReleasesByStatusQuery @Autowired()(implicit val releasePersistence: ReleasePersistence,
                                             implicit val detailsQuery: LiveReleasesQuery,
                                             @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect) extends LiveQuery {

  import LiveReleasesByStatusQuery._

  private val COLUMNS = s"$alias.${RELEASES.STATUS}, COUNT($alias.${RELEASES.CI_UID})"

  private val IN_PROGRESS_STATUSES = Seq(ReleaseStatus.IN_PROGRESS, ReleaseStatus.PAUSED)
  private val FAILED_STATUSES = Seq(ReleaseStatus.FAILING, ReleaseStatus.FAILED)
  private val PLANNED_STATUSES = Seq(ReleaseStatus.PLANNED)

  private val DEFAULT_STATUS_FILTERS = statusFilter(IN_PROGRESS_STATUSES ++ FAILED_STATUSES ++ PLANNED_STATUSES)
  private val IN_PROGRESS_FILTERS = statusFilter(IN_PROGRESS_STATUSES)
  private val FAILED_FILTERS = statusFilter(FAILED_STATUSES)
  private val PLANNED_FILTERS = statusFilter(PLANNED_STATUSES)

  @IsReadOnly
  override def executeSummary(tile: Tile, additionalParameters: JMap[String, Any]): AnyRef = {
    countsQuery(tile)
  }

  @IsReadOnly
  override def executeDetails(tile: Tile, additionalParameters: JMap[String, Any]): AnyRef = {
    val abstractStatus = additionalParameters.getOrDefault("view", IN_PROGRESS).asInstanceOf[String]
    val filters = getFilters(tile, abstractStatus)
    tile.setProperty("filters", filters)

    detailsQuery.executeDetails(tile, additionalParameters)
  }

  private def getFilters(tile: Tile, abstractStatus: String): JList[ReportFilter] = {
    val filters = new JArrayList[ReportFilter]
    filters.addAll(tile.getProperty[JList[ReportFilter]]("filters"))

    filters.add(abstractStatus match {
      case IN_PROGRESS => IN_PROGRESS_FILTERS
      case PLANNED => PLANNED_FILTERS
      case FAILED => FAILED_FILTERS
      case _ => DEFAULT_STATUS_FILTERS
    })
    filters
  }

  private def countsQuery(tile: Tile): AnyRef = {
    val filters = getFilters(tile, null)

    val builder = new ReleasesSqlBuilder()
      .customSelect(COLUMNS)
      .withFilters(filters)
      .groupBy(s"$alias.${RELEASES.STATUS}")

    val (sql, params) = builder.build()
    val counts: Map[ReleaseStatus, Int] = releasePersistence.jdbcTemplate.query(sql, (rs: ResultSet, _: Int) => {
      ReleaseStatus.valueOf(rs.getString(1).toUpperCase()) -> rs.getInt(2)
    }, params: _*).asScala.toMap

    Map(
      PLANNED -> sumStatuses(counts, PLANNED_STATUSES),
      IN_PROGRESS -> sumStatuses(counts, IN_PROGRESS_STATUSES),
      FAILED -> sumStatuses(counts, FAILED_STATUSES)
    ).asJava
  }

  private def sumStatuses(counts: Map[ReleaseStatus, Int], statuses: Seq[ReleaseStatus]): Int =
    statuses.foldLeft(0) { case (total, status) => total + counts.getOrElse(status, 0) }
}

object LiveReleasesByStatusQuery {
  val PLANNED = "planned"
  val IN_PROGRESS = "in_progress"
  val FAILED = "failed"
}

