package com.xebialabs.xlrelease.udm.reporting.repository.sql

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.udm.reporting.repository.DeploymentHistoryFactory._
import com.xebialabs.xlrelease.udm.reporting.repository.DeploymentRepository
import com.xebialabs.xlrelease.udm.reporting.repository.sql.persistence.{DeploymentHistoryPersistence, DeploymentPersistence}
import com.xebialabs.xlrelease.udm.reporting.{Deployment, DeploymentHistory}
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Isolation.READ_COMMITTED
import org.springframework.transaction.annotation.Propagation.REQUIRED
import org.springframework.transaction.annotation.Transactional

@Repository
@Transactional(value = "reportingTransactionManager", propagation = REQUIRED, isolation = READ_COMMITTED, rollbackFor = Array(classOf[Throwable]))
class SqlDeploymentRepository(deploymentPersistence: DeploymentPersistence,
                              deploymentHistoryPersistence: DeploymentHistoryPersistence)
                             (@Qualifier("reportingSqlDialect") implicit val dialect: Dialect) extends DeploymentRepository {


  override def create(deployment: Deployment, deploymentHistory: DeploymentHistory): Unit = {
    deploymentPersistence.insert(deployment)
    val history = Option(deploymentHistory).fold(createDeploymentHistory(deployment))(h => h)
    deploymentHistoryPersistence.insert(history)
  }

  override def update(updated: Deployment, deploymentHistory: DeploymentHistory): Unit = {
    deploymentPersistence.update(updated)
    if (deploymentHistory.getOldStatus != deploymentHistory.getNewStatus) {
      deploymentHistoryPersistence.insert(deploymentHistory)
    }
  }

  override def delete(deploymentId: String): Unit = {
    deploymentPersistence.delete(deploymentId)
  }

  override def exists(deploymentId: String): Boolean = deploymentPersistence.exists(deploymentId)

  override def findById(deploymentId: String): Option[Deployment] = deploymentPersistence.findById(deploymentId)

  override def findByTaskId(taskId: String): Seq[Deployment] = {
    val query = new DeploymentsSqlBuilder()
      .withTaskId(taskId)
      .build()

    deploymentPersistence.findByQuery(query)
  }

  override def findHistoryById(deploymentId: String): Seq[DeploymentHistory] =
    deploymentHistoryPersistence.findById(deploymentId)

  override def findHistoryByIds(deploymentIds: Seq[String]): Map[String, Seq[DeploymentHistory]] =
    deploymentHistoryPersistence.findByIds(deploymentIds)
      .groupBy(_.getDeploymentId)

  override def deleteAutoGeneratedDeployments(taskId: String): Unit =
    deploymentPersistence.deleteAllAutoGenerated(taskId)

}
