package com.xebialabs.xlrelease.udm.reporting.repository.sql.queries

import com.xebialabs.xlrelease.api.v1.forms.ReleaseOrderMode
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import com.xebialabs.xlrelease.reports.filters.{CompositeFilter, ReportFilter}
import com.xebialabs.xlrelease.repository.Ids.normalizeId
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasesSqlBuilder.{alias, folderAlias}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES}
import com.xebialabs.xlrelease.repository.sql.persistence.{ReleasePersistence, ReleasesSqlBuilder}
import com.xebialabs.xlrelease.udm.reporting.filters.impl.ReleaseStatusFilter
import com.xebialabs.xlrelease.udm.reporting.repository.sql.queries.LiveReleasesQuery.{LiveReleasesQueryResult, ReleaseDetails}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.RowMapper
import org.springframework.stereotype.Component

import java.sql.ResultSet
import java.util.{Date, List => JList, Map => JMap}
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._

object LiveReleasesQuery {

  class LiveReleasesQueryResult {
    @BeanProperty
    var total: Int = 0

    @BeanProperty
    var releases: JList[ReleaseDetails] = _
  }

  class ReleaseDetails {
    @BeanProperty
    var releaseId: String = _

    @BeanProperty
    var status: ReleaseStatus = _

    @BeanProperty
    var title: String = _

    @BeanProperty
    var folderId: String = _

    @BeanProperty
    var startDate: Date = _

    @BeanProperty
    var endDate: Date = _
  }

}

@Component
@IsTransactional
class LiveReleasesQuery @Autowired()(implicit val releasePersistence: ReleasePersistence,
                                     @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect) extends LiveQuery {

  private def hasStatusFilter(filters: Seq[ReportFilter]): Boolean = {
    filters.exists({
      case filter: CompositeFilter => hasStatusFilter(filter.getFilters.asScala.toSeq)
      case _: ReleaseStatusFilter => true
      case _ => false
    })
  }

  private def sqlBuilder(filters: JList[ReportFilter], excludeTemplates: Boolean): ReleasesSqlBuilder = {
    val builder = new ReleasesSqlBuilder()
    builder.withFilters(filters)

    if (excludeTemplates) builder.withoutStatuses(Seq(ReleaseStatus.TEMPLATE))

    builder
  }

  private def totalCount(filters: JList[ReportFilter], excludeTemplates: Boolean): Int = {
    val (sql, params) = sqlBuilder(filters, excludeTemplates)
      .customSelect(s"COUNT($alias.${RELEASES.CI_UID})")
      .build()

    releasePersistence.jdbcTemplate.query(sql, params.toArray, (rs: ResultSet, _: Int) => {
      rs.getInt(1)
    }).get(0)
  }

  private def fullFolderId(folderPath: String, folderId: String): String = {
    Seq("Applications", folderPath, folderId)
      .map(normalizeId).filterNot(_.equals("")).mkString("/")
  }

  @IsReadOnly
  override def executeSummary(tile: Tile, additionalParameters: JMap[String, Any]): AnyRef = {
    val limit = Integer.valueOf(additionalParameters.getOrDefault("pageSize", "100").asInstanceOf[String])
    runQuery(tile, limit, 0)
  }

  @IsReadOnly
  override def executeDetails(tile: Tile, additionalParameters: JMap[String, Any]): AnyRef = {
    val limit = Integer.valueOf(additionalParameters.getOrDefault("pageSize", "100").asInstanceOf[String])
    val offset = Integer.valueOf(additionalParameters.getOrDefault("offset", "0").asInstanceOf[String])

    runQuery(tile, limit, offset)
  }

  private def runQuery(tile: Tile, limit: Int, offset: Int): AnyRef = {
    val filters = tile.getProperty[JList[ReportFilter]]("filters")

    val excludeTemplates = !hasStatusFilter(filters.asScala.toSeq)
    val (sql, params) = sqlBuilder(filters, excludeTemplates)
      .customSelect(
        s"$alias.${RELEASES.STATUS}",
        s"$folderAlias.${FOLDERS.FOLDER_ID}",
        s"$folderAlias.${FOLDERS.FOLDER_PATH}",
        s"$alias.${RELEASES.RELEASE_TITLE}",
        s"$alias.${RELEASES.START_DATE}",
        s"$alias.${RELEASES.END_DATE}",
        s"$alias.${RELEASES.RELEASE_ID}"
      )
      .orderBy(ReleaseOrderMode.start_date)
      .addFolderJoin()
      .limitAndOffset(limit, offset)
      .build()

    val result = new LiveReleasesQueryResult
    result.releases = releasePersistence.jdbcTemplate.query[ReleaseDetails](sql, params.toArray, releaseRowMapper)
    result.total = if (offset == 0 && limit > result.releases.size) {
      result.releases.size
    } else {
      totalCount(filters, excludeTemplates)
    }

    result
  }

  private val releaseRowMapper: RowMapper[ReleaseDetails] = (rs: ResultSet, _: Int) => {
    val folderId = rs.getString(FOLDERS.FOLDER_ID)
    val parentPath = rs.getString(FOLDERS.FOLDER_PATH)

    val release = new ReleaseDetails
    release.status = ReleaseStatus.valueOf(rs.getString(RELEASES.STATUS).toUpperCase())
    release.folderId = fullFolderId(parentPath, folderId)
    release.title = rs.getString(RELEASES.RELEASE_TITLE)
    release.startDate = rs.getTimestamp(RELEASES.START_DATE)
    release.endDate = rs.getTimestamp(RELEASES.END_DATE)
    release.releaseId = rs.getString(RELEASES.RELEASE_ID)
    release
  }
}
