package com.xebialabs.xlrelease.udm.reporting.repository.sql

import com.xebialabs.xlrelease.api.v1.forms.TimeFrame
import com.xebialabs.xlrelease.db.ArchivedReleases.{COMMON_END_DATE_COLUMN, COMMON_START_DATE_COLUMN, shortenId}
import com.xebialabs.xlrelease.db.sql.Sql
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.archiving.BaseSelectArchivedBuilder
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.{getFolderlessId, releaseIdFrom}
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence.hash
import com.xebialabs.xlrelease.udm.reporting.filters._
import com.xebialabs.xlrelease.udm.reporting.repository.sql.persistence.DeploymentSchema

import java.util.Date

class DeploymentsSqlBuilder(implicit dialect: Dialect)
  extends BaseSelectArchivedBuilder[DeploymentsSqlBuilder]("DEPLOYMENTS", "deployments", "*") {

  def withTaskId(taskId: String): DeploymentsSqlBuilder = {
    conditions += Sql(
      s"""$alias.${DeploymentSchema.DEPLOYMENTS.RELEASE_ID} = ?
         |AND $alias.${DeploymentSchema.DEPLOYMENTS.TASK_ID_HASH} = ?
         |AND $alias.${DeploymentSchema.DEPLOYMENTS.TASK_ID} = ?
       """.stripMargin, Seq(
        shortenId(releaseIdFrom(taskId)),
        hash(taskId), getFolderlessId(taskId))
    )
    this
  }

  def withEnvironmentId(environmentId: String): DeploymentsSqlBuilder = {
    conditions += Sql(s"$alias.${DeploymentSchema.DEPLOYMENTS.ENVIRONMENT_ID} = ?", Seq(Ids.getFolderlessId(environmentId)))
    this
  }

  def withEnvironmentIds(environmentIds: Seq[String]): DeploymentsSqlBuilder = {
    if (environmentIds.nonEmpty) {
      val expression = environmentIds
        .grouped(500)
        .map(pagedIds => s"$alias.${DeploymentSchema.DEPLOYMENTS.ENVIRONMENT_ID} IN (${pagedIds.map(_ => "?").mkString(", ")})")
        .mkString("(", " OR \n", ")")
      conditions += Sql(expression, environmentIds)
    }
    this
  }

  def withApplicationId(applicationId: String): DeploymentsSqlBuilder = {
    conditions += Sql(s"$alias.${DeploymentSchema.DEPLOYMENTS.APPLICATION_ID} = ?", Seq(Ids.getFolderlessId(applicationId)))
    this
  }

  def withFilters(filters: Seq[ReportFilter]): DeploymentsSqlBuilder = {
    if (filters != null) {
      val visitor = new DeploymentsSqlBuilderVisitor(this)
      filters.foreach {
        case udmReportFilter: UdmReportFilter => udmReportFilter.accept(visitor)
      }
    }
    this
  }

  val START_DATE_EXPRESSION = s"""(? <= $alias.$COMMON_END_DATE_COLUMN OR ($alias.$COMMON_END_DATE_COLUMN IS NULL AND ? <= $alias.$COMMON_START_DATE_COLUMN))"""

  override def withDates(timeFrame: TimeFrame, from: Date, to: Date): DeploymentsSqlBuilder = {
    val startDate = new Date(timeFrame.getStartDate(from))
    val endDate = new Date(timeFrame.getEndDate(to))
    if (startDate != null && endDate != null) {
      conditions += Sql(s"($START_DATE_EXPRESSION AND ? > $alias.$COMMON_START_DATE_COLUMN)", Seq(startDate, startDate, endDate))
    } else {
      withStartDate(startDate)
      withEndDate(endDate)
    }
    this
  }

  override def withStartDate(startDate: Date): DeploymentsSqlBuilder = {
    if (startDate != null) {
      conditions += Sql(START_DATE_EXPRESSION, Seq(startDate, startDate))
    }
    this
  }

  override def newInstance: DeploymentsSqlBuilder = new DeploymentsSqlBuilder()

}
