package com.xebialabs.xlrelease.udm.reporting.events

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.filter.{ApplicationFilters, EnvironmentFilters}
import com.xebialabs.xlrelease.domain.environments.{Application, Environment}
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.udm.reporting.DeploymentStatus
import com.xebialabs.xlrelease.domain.{Release, Task, VisitableItem}
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentRepository}
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.service.{FolderVariableService, VariableService}
import com.xebialabs.xlrelease.udm.reporting._
import com.xebialabs.xlrelease.udm.reporting.repository.DeploymentHistoryFactory._
import com.xebialabs.xlrelease.udm.reporting.repository.DeploymentRepository
import com.xebialabs.xlrelease.variable.VariableResolver
import grizzled.slf4j.Logging
import org.springframework.util.StringUtils.hasText

import scala.util.Try

object DeploymentListener {
  val ENVIRONMENT_REGEX: String = ".*Environment[0-9a-f]{32}"
  val APPLICATION_REGEX: String = ".*Application[0-9a-f]{32}"
  val UNKNOWN_TARGET = "Unknown"
  val DEFAULT_PAGE = Page(0, 100, 0)
}

import scala.jdk.CollectionConverters._

trait DeploymentListener extends Logging {
  val applicationRepository: ApplicationRepository
  val environmentRepository: EnvironmentRepository
  val variableService: VariableService
  val folderVariableService: FolderVariableService
  val deploymentRepository: DeploymentRepository

  import DeploymentListener._

  def saveDeployment(task: Task, facet: DeploymentTaskFacet, status: DeploymentStatus, autoGenerated: Boolean): Unit = {
    logger.trace(s"Saving deployment event for facet $facet")

    val previous = deploymentRepository.findById(getDeploymentId(facet))
    val updated = convertToDeployment(task, facet, status, autoGenerated, previous)
    logger.debug(s"Saving deployment event $updated")
    if (previous.isEmpty) {
      deploymentRepository.create(updated, createDeploymentHistory(updated, task))
    } else {
      deploymentRepository.update(updated, createDeploymentHistory(updated, task, previous))
    }

  }

  def calculateDeploymentStatus(task: Task): Option[DeploymentStatus] = {
    if (task.getStatus == TaskStatus.COMPLETED) {
      Some(DeploymentStatus.COMPLETED)
    } else if (task.getStatus == TaskStatus.ABORTED) {
      Some(DeploymentStatus.ABORTED)
    } else if (task.getStatus == TaskStatus.SKIPPED) {
      Some(DeploymentStatus.SKIPPED)
    } else if (task.getStatus.isOneOf(TaskStatus.FAILING, TaskStatus.FAILED)) {
      Some(DeploymentStatus.FAILED)
    } else if (task.getStatus.isOneOf(TaskStatus.PLANNED, TaskStatus.COMPLETED_IN_ADVANCE, TaskStatus.SKIPPED_IN_ADVANCE)) {
      None
    } else {
      Some(DeploymentStatus.IN_PROGRESS)
    }
  }

  def convertToDeployment(task: Task, facet: DeploymentTaskFacet, status: DeploymentStatus,
                          autoGenerated: Boolean, previousDeployment: Option[Deployment]): Deployment = {
    val deployment = new Deployment
    deployment.setDeploymentId(getDeploymentId(facet))
    deployment.setStatus(status)

    deployment.setTaskId(task.getId)
    deployment.setTaskTitle(task.getTitle)
    deployment.setTaskType(task.getTaskType.toString)
    val owner = if (task.isAutomated && !hasText(task.getOwner)) {
      task.getRelease.getScriptUsername
    } else {
      task.getOwner
    }
    deployment.setTaskOwner(owner)
    deployment.setTaskTeam(task.getTeam)
    val failuresCount = previousDeployment.map(_.getFailuresCount).getOrElse(0) + (if (hasJustFailed(status, previousDeployment)) 1 else 0)
    deployment.setFailuresCount(failuresCount)
    deployment.setAutomated(task.isAutomated)

    deployment.setReleaseId(task.getRelease.getId)
    deployment.setReleaseTitle(task.getRelease.getTitle)
    deployment.setReleaseOwner(task.getReleaseOwner)

    deployment.setSourceId(if (autoGenerated) null else facet.getId)
    deployment.setStartDate(task.getStartDate)
    deployment.setEndDate(task.getEndDate)
    if (deployment.getEndDate != null && task.getStartDate != null) {
      deployment.setDuration(((deployment.getEndDate.getTime - deployment.getStartDate.getTime) / 1000).intValue())
    }

    val environment = resolveEnvironment(facet.getEnvironmentId)
    deployment.setEnvironmentId(environment.getId)
    deployment.setEnvironmentName(if (hasText(environment.getTitle)) environment.getTitle else UNKNOWN_TARGET)

    val application = resolveApplication(facet.getApplicationId)
    deployment.setApplicationId(application.getId)
    deployment.setApplicationName(if (hasText(application.getTitle)) application.getTitle else UNKNOWN_TARGET)

    deployment.setVersion(if (hasText(facet.getVersion)) facet.getVersion else UNKNOWN_TARGET)
    deployment
  }

  private def hasJustFailed(status: DeploymentStatus, previousDeployment: Option[Deployment]): Boolean = {
    status == DeploymentStatus.FAILED && previousDeployment.map(_.getStatus).getOrElse(DeploymentStatus.PLANNED) != status
  }

  def getDeploymentId(facet: Facet): String = {
    facet.getId.replace("Facet", "Deployment")
  }

  def resolveVariables(item: VisitableItem, release: Release): Set[String] = {
    val globalVariables = variableService.findGlobalVariablesOrEmpty().getVariables.asScala.toSeq
    val folderVariables = folderVariableService.getAllFromAncestry(release.findFolderId()).getVariables.asScala.toSeq
    val releaseVariables = release.getAllVariables.asScala.toSeq
    VariableResolver.resolve(item, releaseVariables ++ globalVariables ++ folderVariables)._1
  }

  def resolveEnvironment(environmentIdOrName: String): Environment = (Option(environmentIdOrName) match {
    case None => None
    case Some(id) if id.matches(ENVIRONMENT_REGEX) =>
      Try(environmentRepository.findEnvironmentById(id)).toOption
    case Some(name) =>
      val filters = new EnvironmentFilters
      filters.setTitle(name)
      environmentRepository.searchEnvironments(filters, DEFAULT_PAGE).find(_.getTitle == name)
    case _ => None
  }).getOrElse(getUnknownEnvironment(environmentIdOrName))

  def getUnknownEnvironment(title: String): Environment = {
    val environment = new Environment
    environment.setId(null)
    environment.setTitle(title)
    environment
  }

  def resolveApplication(applicationIdOrName: String): Application = (Option(applicationIdOrName) match {
    case None => None
    case Some(id) if id.matches(APPLICATION_REGEX) =>
      Try(applicationRepository.findApplicationById(id)).toOption
    case Some(name) =>
      val filters = new ApplicationFilters()
      filters.setTitle(name)
      applicationRepository.searchApplications(filters, DEFAULT_PAGE).find(_.getTitle == name)
    case _ => None
  }).getOrElse(getUnknownApplication(applicationIdOrName))

  def getUnknownApplication(title: String): Application = {
    val application = new Application
    application.setId(null)
    application.setTitle(title)
    application
  }

  def isDeploymentFacet(facet: Facet): Boolean = facet.getType.instanceOf(Type.valueOf(classOf[DeploymentTaskFacet]))
}

