import React from 'react';
import { DotAutoComplete } from '@digital-ai/dot-components';

import { mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { DeploymentTaskFacetCommon, DeploymentTaskFacetCommonProps } from './deployment-task-facet-common.component';
import { stringPropertyMock } from '../../../../../../../../../../core/xlr-ui/app/features/tasks/__mocks__/property.mock';
import { WithVariables } from '../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/config/components/task-variables/with-variables.component';

describe('DeploymentTaskFacetCommon', () => {
    const dispatch = jest.fn();
    const onInit = jest.fn();
    const onChange = jest.fn();
    const onNewVariable = jest.fn();
    let wrapper: ReactWrapper;

    const defaultProps: DeploymentTaskFacetCommonProps = {
        onInit: onInit as never,
        onChange,
        disabled: false,
        facet: { id: 'Facet1', type: 'udm.Build', targetId: 'Task1', variableMapping: {}, string: 'Application2' },
        onNewVariable,
        property: stringPropertyMock,
        options: [
            { id: 'Application1', title: 'App1' },
            { id: 'Application2', title: 'App2' },
        ],
    };

    const getWithVariables = () => wrapper.find(WithVariables);
    const getDotAutoComplete = () => wrapper.find('.deployment-task-facet-common').find(DotAutoComplete);

    beforeEach(() => {
        dispatch.mockReset();
        onChange.mockReset();
        wrapper = mountWithStoreAndTheme(<DeploymentTaskFacetCommon {...defaultProps} />, dispatch, { taskDrawer: { allVariables: [] } });
    });

    it('should trigger onInit', () => {
        expect(onInit).toHaveBeenCalled();
    });

    it('should create WithVariables with proper props', () => {
        const props = getWithVariables().props();
        expect(props.disabled).toStrictEqual(defaultProps.disabled);
        expect(props.displayType).toStrictEqual('Text');
        expect(props.helperText).toStrictEqual(defaultProps.property.description);
        expect(props.inputId).toStrictEqual(defaultProps.property.fqn);
        expect(props.isRequired).toStrictEqual(defaultProps.property.required);
        expect(props.label).toStrictEqual(defaultProps.property.label);
        expect(props.variableOrValue).toStrictEqual({
            value: 'Application2',
            variable: undefined,
        });
        expect(props.variableType).toStrictEqual('xlrelease.StringVariable');
        expect(props.variables).toStrictEqual([]);
    });

    it('should createDotAutoComplete with proper props', () => {
        const props = getDotAutoComplete().props();
        expect(props.freesolo).toStrictEqual(false);
        expect(props.helperText).toStrictEqual(defaultProps.property.description);
        expect(props.inputId).toStrictEqual(defaultProps.property.fqn);
        expect(props.label).toStrictEqual(defaultProps.property.label);
        expect(props.multiple).toStrictEqual(false);
        expect(props.options).toStrictEqual(defaultProps.options);
        expect(props.readOnly).toStrictEqual(defaultProps.disabled);
        expect(props.required).toStrictEqual(defaultProps.property.required);
        expect(props.value).toStrictEqual({ id: 'Application2', title: 'App2' });
    });

    it('should have null for DotAutoComplete value prop', () => {
        const testProps = {
            ...defaultProps,
            options: [{ id: 'Application1', title: 'App1' }],
        };
        wrapper = mountWithStoreAndTheme(<DeploymentTaskFacetCommon {...testProps} />, dispatch, { taskDrawer: { allVariables: [] } });
        const props = getDotAutoComplete().props();
        expect(props.value).toStrictEqual(null);
    });

    it('should handle when creating new variable', () => {
        const props = getWithVariables().props();
        props.onNewVariableAdd('var', 'xlrelease.StringVariable');
        expect(onNewVariable).toHaveBeenCalledWith('var', 'xlrelease.StringVariable');
    });

    it('should handle when variable has changed', () => {
        const props = getWithVariables().props();
        props.onChange({ variable: '${var1}', value: 'whatever' });
        expect(onChange).toHaveBeenCalledWith('${var1}');
    });

    it('should handle when value has changed', () => {
        const props = getDotAutoComplete().props();
        props.onChange?.('' as never, { id: 'Application1', title: 'App1' } as never, '');
        expect(onChange).toHaveBeenCalledWith('Application1');
    });

    it('should handle onChange when selected value is null', () => {
        const props = getDotAutoComplete().props();
        props.onChange?.('' as never, null, '');
        expect(onChange).toHaveBeenCalledWith(null);
    });
});
