package com.xebialabs.xlrelease.api.v1.impl;

import java.util.List;
import java.util.stream.Collectors;
import jakarta.ws.rs.core.UriInfo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.api.v1.TriggersApi;
import com.xebialabs.xlrelease.api.v1.filter.TriggerFilters;
import com.xebialabs.xlrelease.domain.ReleaseTrigger;
import com.xebialabs.xlrelease.domain.Trigger;
import com.xebialabs.xlrelease.triggers.security.TriggerPermissionChecker;
import com.xebialabs.xlrelease.triggers.service.TriggerService;
import com.xebialabs.xlrelease.utils.SortSupport;
import com.xebialabs.xlrelease.view.ReleaseTriggerOverview;
import com.xebialabs.xlrelease.views.BulkActionResultView;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.deployit.checks.Checks.checkArgument;

@Controller
public class TriggersApiImpl implements TriggersApi {

    private final TriggerPermissionChecker triggerPermissionChecker;
    private final TriggerService triggerService;

    @Autowired
    public TriggersApiImpl(TriggerPermissionChecker triggerPermissionChecker, TriggerService triggerService) {
        this.triggerPermissionChecker = triggerPermissionChecker;
        this.triggerService = triggerService;
    }

    @Timed
    @Override
    public Page<ReleaseTriggerOverview> searchTriggers(TriggerFilters triggerFilters,
                                                       int page,
                                                       int resultsPerPage,
                                                       UriInfo uriInfo) {
        checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than %d", DEFAULT_RESULTS_PER_PAGE);
        PageRequest pageable = PageRequest.of(page, resultsPerPage, SortSupport.toSort(uriInfo));
        return searchTriggers(triggerFilters, pageable);
    }

    @Timed
    @Override
    public List<ReleaseTriggerOverview> searchTriggers(TriggerFilters triggerFilters, int page, int resultsPerPage, Sort sort) {
        PageRequest pageable = PageRequest.of(page, resultsPerPage, sort);
        return searchTriggers(triggerFilters, pageable).getContent();
    }

    @Timed
    @Override
    public List<ReleaseTriggerOverview> searchTriggers(TriggerFilters triggerFilters, int page, int resultsPerPage) {
        return searchTriggers(triggerFilters, page, resultsPerPage, Sort.unsorted());
    }

    private Page<ReleaseTriggerOverview> searchTriggers(TriggerFilters triggerFilters, Pageable pageable) {
        if (triggerFilters == null) {
            triggerFilters = new TriggerFilters();
        }
        return triggerService.findBy(triggerFilters, pageable);
    }

    @Timed
    @Override
    public void enableTrigger(String triggerId) {
        triggerPermissionChecker.checkEditTrigger(triggerId);
        triggerService.updateTriggerStatus(triggerId, true, true);
    }

    @Timed
    @Override
    public BulkActionResultView enableTriggers(List<String> triggerIds) {
        List<String> allowedTriggers = triggerPermissionChecker.filterEditableTriggers(triggerIds);
        List<String> enabledTriggers = triggerService.enableTriggers(allowedTriggers);
        return new BulkActionResultView(enabledTriggers);
    }

    @Timed
    @Override
    public BulkActionResultView enableAllTriggers() {
        List<String> enabledTriggers = triggerService.enableAllAccessibleTriggers();
        return new BulkActionResultView(enabledTriggers);
    }

    @Timed
    @Override
    public void disableTrigger(String triggerId) {
        triggerPermissionChecker.checkEditTrigger(triggerId);
        triggerService.updateTriggerStatus(triggerId, false, true);
    }

    @Timed
    @Override
    public BulkActionResultView disableTriggers(List<String> triggerIds) {
        List<String> allowedTriggers = triggerPermissionChecker.filterEditableTriggers(triggerIds);
        List<String> disabledTriggers = triggerService.disableTriggers(allowedTriggers);
        return new BulkActionResultView(disabledTriggers);
    }

    @Timed
    @Override
    public BulkActionResultView disableAllTriggers() {
        List<String> disabledTriggers = triggerService.disableAllAccessibleTriggers();
        return new BulkActionResultView(disabledTriggers);
    }

    @Timed
    @Override
    public void removeTrigger(String triggerId) {
        triggerPermissionChecker.checkEditTrigger(triggerId);
        triggerService.deleteTrigger(triggerId);
    }

    @Timed
    @Override
    public void runTrigger(String triggerId) {
        triggerPermissionChecker.checkEditTrigger(triggerId);
        triggerService.runNowTrigger(triggerId);
    }

    @Timed
    @Override
    public List<Descriptor> getTypes() {
        return DescriptorRegistry.getSubtypes(Type.valueOf(Trigger.class)).stream()
                .map(Type::getDescriptor)
                .filter(
                        desc -> !desc.isVirtual() &&
                                !desc.getType().equals(Type.valueOf(ReleaseTrigger.class)))
                .collect(Collectors.toList());
    }

    @Timed
    @Override
    public Trigger addTrigger(Trigger trigger) {
        triggerPermissionChecker.checkEditTrigger(trigger);
        return triggerService.addTrigger(trigger);
    }

    @Timed
    @Override
    public Trigger getTrigger(String triggerId) {
        Trigger trigger = triggerService.findById(triggerId);
        triggerPermissionChecker.checkViewTrigger(trigger);
        return trigger;
    }

    @Timed
    @Override
    public Trigger updateTrigger(String triggerId, Trigger trigger) {
        triggerPermissionChecker.checkEditTrigger(triggerId);
        trigger.setId(triggerId);
        return triggerService.updateTrigger(trigger);
    }

}
