package com.xebialabs.xlrelease.triggers.repository.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.{Release, ReleaseTrigger, Trigger}
import com.xebialabs.xlrelease.limits.LimitEnforcer
import com.xebialabs.xlrelease.limits.LimitEnforcer.LimitType
import com.xebialabs.xlrelease.repository.sql.ConfigurationReferencesSupport
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationReferencePersistence
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport, ReleasePersistence}
import com.xebialabs.xlrelease.repository.{Ids, PersistenceInterceptor}
import com.xebialabs.xlrelease.triggers.deployment_based.StatusWebhookTrigger
import com.xebialabs.xlrelease.triggers.event_based.EventBasedTrigger
import com.xebialabs.xlrelease.utils.TenantContext
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.JdbcTemplate

@IsTransactional
class TriggerPersistenceInterceptor(val triggerPersistence: TriggerPersistence,
                                    val releasePersistence: ReleasePersistence,
                                    val configurationPersistence: ConfigurationReferencePersistence,
                                    val jdbcTemplate: JdbcTemplate,
                                    val dialect: Dialect,
                                    limitEnforcer: LimitEnforcer)
  extends PersistenceInterceptor[Trigger] with Logging with ReleaseTriggerPersistence with ConfigurationReferencesSupport {

  override def getReleasePersistence: ReleasePersistence = releasePersistence

  override def onCreate(ci: Trigger): Unit = {
    val tenantId = TenantContext.getTenant()
    limitEnforcer.enforceLimit(tenantId, LimitType.TRIGGERS, 1, () => triggerPersistence.tenantTriggerCount(tenantId))
  }

  override def afterCreate(ci: Trigger): Unit = {
    logger.info(s"Trigger created: '${ci.getType}' ciUid:'${ci.getCiUid}' id: '${ci.getId}'")
    updateConfigurationRefs(ci)
    ci match {
      case rt: ReleaseTrigger => link(rt)
      case rt: EventBasedTrigger => link(rt)
      case rt: StatusWebhookTrigger => link(rt)
      case _ => // nothing
    }
  }

  override def afterUpdate(ci: Trigger): Unit = {
    updateConfigurationRefs(ci)
    ci match {
      case rt: ReleaseTrigger => updateLink(rt)
      case rt: EventBasedTrigger => updateLink(rt)
      case rt: StatusWebhookTrigger => updateLink(rt)
      case _ => // nothing
    }
  }

  override def onDelete(triggerId: String): Unit = {
    val trigger = triggerPersistence.findById[Trigger](Ids.getFolderlessId(triggerId.normalized))
    trigger match {
      case Some(rt: ReleaseTrigger) => unlink(rt.getCiUid)
      case Some(rt: EventBasedTrigger) => unlink(rt.getCiUid)
      case Some(rt: StatusWebhookTrigger) => unlink(rt.getCiUid)
      case _ => // nothing
    }
  }

  private def unlink(triggerCiUid: CiUid): Unit = {
    sqlUpdate(
      s"DELETE FROM ${TEMPLATE_TRIGGERS.TABLE} WHERE ${TEMPLATE_TRIGGERS.TRIGGER_UID} = :${TEMPLATE_TRIGGERS.TRIGGER_UID}",
      paramSource(TEMPLATE_TRIGGERS.TRIGGER_UID -> triggerCiUid),
      numOfRows => logger.info(s"DELETED $numOfRows from ${TEMPLATE_TRIGGERS.TABLE}")
    )
  }
}

/**
 * Intercepts Template CRUD operations.
 * Main purpose is to handle Trigger lifecycle.
 *
 * @param jdbcTemplate
 * @param dialect
 */
@IsTransactional
class TriggerTemplatePersistenceInterceptor(val releasePersistence: ReleasePersistence,
                                            val triggerPersistence: TriggerPersistence,
                                            val jdbcTemplate: JdbcTemplate,
                                            val dialect: Dialect)
  extends PersistenceInterceptor[Release] with Logging with ReleaseTriggerPersistence {

  val NL: String = System.lineSeparator()

  override def getReleasePersistence: ReleasePersistence = releasePersistence

  override def onDelete(templateId: String): Unit = {
    unlink(templateId)
  }

  private[repository] def unlink(templateId: String): Unit = {
    val releaseCiUid: Integer = releasePersistence.findUidByReleaseId(templateId).get
    sqlUpdate(
      s"DELETE FROM ${TEMPLATE_TRIGGERS.TABLE} WHERE ${TEMPLATE_TRIGGERS.RELEASE_UID} = :${TEMPLATE_TRIGGERS.RELEASE_UID}",
      paramSource(TEMPLATE_TRIGGERS.RELEASE_UID -> releaseCiUid),
      _ => ()
    )
  }

}


trait ReleaseTriggerPersistence extends PersistenceSupport with Logging {
  def getReleasePersistence: ReleasePersistence

  def releasePersistence(): ReleasePersistence

  private def insertTemplateTrigger(releaseCiUid: CiUid, triggerCiUid: CiUid): Unit =
    sqlInsert(s"INSERT INTO ${TEMPLATE_TRIGGERS.TABLE} VALUES(:${TEMPLATE_TRIGGERS.RELEASE_UID}, :${TEMPLATE_TRIGGERS.TRIGGER_UID})",
      paramSource(
        TEMPLATE_TRIGGERS.RELEASE_UID -> releaseCiUid,
        TEMPLATE_TRIGGERS.TRIGGER_UID -> triggerCiUid
      )
    )

  private[repository] def link(releaseTrigger: ReleaseTrigger): Unit = {
    val releaseCiUid = releasePersistence().findUidByReleaseId(releaseTrigger.getTemplate).get
    insertTemplateTrigger(releaseCiUid, releaseTrigger.getCiUid)
  }

  private[repository] def link(eventBasedTrigger: EventBasedTrigger): Unit =
    eventBasedTrigger.getTemplateId.flatMap(getReleasePersistence.findUidByReleaseId).foreach(templateCiUid => {
      insertTemplateTrigger(templateCiUid, eventBasedTrigger.getCiUid)
    })

  private[repository] def link(deploymentTrigger: StatusWebhookTrigger): Unit =
    deploymentTrigger.getTemplateId.flatMap(getReleasePersistence.findUidByReleaseId).foreach(templateCiUid => {
      insertTemplateTrigger(templateCiUid, deploymentTrigger.getCiUid)
    })

  private val STMT_UPDATE_TEMPLATE_TRIGGERS =
    s"""UPDATE ${TEMPLATE_TRIGGERS.TABLE}
       | SET ${TEMPLATE_TRIGGERS.RELEASE_UID} = :${TEMPLATE_TRIGGERS.RELEASE_UID}
       | WHERE ${TEMPLATE_TRIGGERS.TRIGGER_UID} = :${TEMPLATE_TRIGGERS.TRIGGER_UID}
       |""".stripMargin

  private def updateTemplateTrigger(templateCiUid: CiUid, triggerCiUid: CiUid): Unit = {
    sqlUpdate(STMT_UPDATE_TEMPLATE_TRIGGERS, paramSource(
      TEMPLATE_TRIGGERS.RELEASE_UID -> templateCiUid,
      TEMPLATE_TRIGGERS.TRIGGER_UID -> triggerCiUid
    ), _ => ())
  }

  private[repository] def updateLink(releaseTrigger: ReleaseTrigger): Unit = {
    val releaseCiUid = releasePersistence().findUidByReleaseId(releaseTrigger.getTemplate).get
    updateTemplateTrigger(releaseCiUid, releaseTrigger.getCiUid)
  }

  private[repository] def updateLink(eventBasedTrigger: EventBasedTrigger): Unit = {
    eventBasedTrigger.getTemplateId.flatMap(getReleasePersistence.findUidByReleaseId).foreach(templateCiUid => {
      updateTemplateTrigger(templateCiUid, eventBasedTrigger.getCiUid)
    })
  }

  private[repository] def updateLink(deploymentTrigger: StatusWebhookTrigger): Unit = {
    deploymentTrigger.getTemplateId.flatMap(getReleasePersistence.findUidByReleaseId).foreach(templateCiUid => {
      updateTemplateTrigger(templateCiUid, deploymentTrigger.getCiUid)
    })
  }
}
