package com.xebialabs.xlrelease.triggers.config

import com.xebialabs.xlrelease.actors.{ActorSystemHolder, ManagedActor, ReleaseActorService}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration
import com.xebialabs.xlrelease.script.ScriptVariables
import com.xebialabs.xlrelease.script.jython.JythonScriptService
import com.xebialabs.xlrelease.service.{ConfigurationVariableService, ReleaseService}
import com.xebialabs.xlrelease.spring.config.{SqlConfiguration, SqlRepositoriesConfiguration}
import com.xebialabs.xlrelease.triggers.actors.TriggerActor.TriggerAction
import com.xebialabs.xlrelease.triggers.actors.{DefaultTriggerLifecycle, TriggerActor, TriggerIdExtensions, TriggerOperations, TriggerProcessingActor}
import com.xebialabs.xlrelease.triggers.config.TriggerExecutionConfiguration.TriggerActorHolder
import com.xebialabs.xlrelease.triggers.deployment_based.StatusWebhookTriggerLifecycle
import com.xebialabs.xlrelease.triggers.event_based.EventBasedTriggerLifecycle
import com.xebialabs.xlrelease.triggers.events.handler.TriggerEventHandler
import com.xebialabs.xlrelease.triggers.scheduled.quartz.QuartzScheduledJobService
import com.xebialabs.xlrelease.triggers.scheduled.validators.ScheduledTriggerValidator
import com.xebialabs.xlrelease.triggers.scheduled.{ReleaseTriggerLifecycle, ScheduledJobService, TriggerScriptService}
import com.xebialabs.xlrelease.triggers.service.TriggerService
import com.xebialabs.xlrelease.triggers.service.impl._
import com.xebialabs.xlrelease.webhooks.mapping.PropertiesMapper
import com.xebialabs.xlrelease.webhooks.registry.SubscriberRegistry
import org.apache.pekko.cluster.sharding.ShardRegion
import org.springframework.context.ApplicationContext
import org.springframework.context.annotation._

object TriggerExecutionConfiguration {
  type TriggerActorHolder = ManagedActor[TriggerActor]
}

@Configuration
class TriggerExecutionConfiguration(applicationContext: ApplicationContext,
                                    sqlConfiguration: SqlConfiguration,
                                    triggerConfiguration: TriggerConfiguration,
                                    quartzConfiguration: QuartzConfiguration,
                                    sqlRepositories: SqlRepositoriesConfiguration,
                                    xlrConfig: XlrConfig,
                                    eventBus: XLReleaseEventBus,
                                    systemHolder: ActorSystemHolder,
                                    jythonScriptService: JythonScriptService,
                                    releaseActorService: ReleaseActorService,
                                    releaseService: ReleaseService,
                                    subscriberRegistry: SubscriberRegistry,
                                    propertiesMapper: PropertiesMapper,
                                    scriptVariables: ScriptVariables,
                                    configurationVariableService: ConfigurationVariableService) {

  /*
    There is (or should be) certain order between classes here:
    - endpoints (http, event handlers) call service
    - service calls actor
    - actor uses operations
    - operations use "lifecycle"
    - at the bottom of the chain are repositories
    If you are not careful you might introduce circular dependency between objects.
   */

  @Bean
  def triggerScriptService(): TriggerScriptService = {
    new TriggerScriptService(jythonScriptService, scriptVariables)
  }

  @Primary
  @Bean
  def triggerLifecycle(): DefaultTriggerLifecycle = {
    new DefaultTriggerLifecycle(List(releaseTriggerLifecycle(), eventBasedTriggerLifecycle(), statusWebhookTriggerLifecycle()))
  }

  @Bean
  def releaseTriggerLifecycle(): ReleaseTriggerLifecycle = {
    new ReleaseTriggerLifecycle(scheduledJobService(),
      triggerScriptService(),
      releaseActorService,
      releaseService,
      eventBus,
      triggerConfiguration.triggerRepository(),
      sqlRepositories.releaseRepository(),
      triggerActorHolder()
    )
  }

  @Bean
  def eventBasedTriggerLifecycle(): EventBasedTriggerLifecycle = {
    new EventBasedTriggerLifecycle(applicationContext,
      subscriberRegistry,
      propertiesMapper,
      eventBus,
      triggerConfiguration.triggerRepository(),
      sqlRepositories.releaseRepository(),
      xlrConfig
    )
  }

  @Bean
  def statusWebhookTriggerLifecycle(): StatusWebhookTriggerLifecycle = {
    new StatusWebhookTriggerLifecycle(applicationContext,
      subscriberRegistry,
      propertiesMapper,
      eventBus,
      triggerConfiguration.triggerRepository(),
      sqlRepositories.releaseRepository(),
      xlrConfig
    )
  }

  @Bean
  def scheduledJobService(): ScheduledJobService = {
    new QuartzScheduledJobService(quartzConfiguration.triggerSchedulerService())
  }

  @Bean
  def scheduledTriggerValidator(): ScheduledTriggerValidator.type = ScheduledTriggerValidator

  @Bean
  @Description("TriggerActorHolder is opaque reference that shields internal ActorRef. " +
    "Actor should call trigger operations." +
    "Actor should be called by TriggerService.")
  def triggerActorHolder(): TriggerActorHolder = {
    if (xlrConfig.isClusterEnabled) {
      val numberOfShards = xlrConfig.sharding.numberOfReleaseShards

      def extractShardId(id: String): String = (math.abs(id.triggerActorName.hashCode) % numberOfShards).toString

      val extractTriggerId: ShardRegion.ExtractEntityId = {
        case msg: TriggerAction => (msg.triggerId.triggerActorName, msg)
      }
      val extractTriggerShardId: ShardRegion.ExtractShardId = {
        case msg: TriggerAction => extractShardId(msg.triggerId)
        case ShardRegion.StartEntity(id) => extractShardId(id)
      }

      systemHolder.shardedActorOf[TriggerActor](
        actorProps = systemHolder.props(classOf[TriggerActor]),
        typeName = "Trigger",
        extractEntityId = extractTriggerId,
        extractShardId = extractTriggerShardId,
      )
    } else {
      systemHolder.actorOf[TriggerActor](classOf[TriggerProcessingActor], TriggerProcessingActor.name)
    }
  }

  @Bean
  def triggerOperations(): TriggerOperations = {
    new TriggerOperations(
      triggerConfiguration.triggerRepository(),
      sqlRepositories.releaseRepository(),
      eventBus,
      triggerLifecycle(),
      configurationVariableService
    )
  }

  @Bean
  @Description("TriggerService should not be injected into TriggerActor, TriggerOperations or TriggerLifecycle.")
  def triggerService(): TriggerService = {
    new DefaultTriggerService(
      triggerActorHolder(),
      sqlConfiguration.ciIdService(),
      triggerConfiguration.triggerRepository(),
      triggerConfiguration.triggerManagementRepository())
  }

  @Bean
  def triggerEventHandler(): TriggerEventHandler = {
    new TriggerEventHandler(
      triggerConfiguration.triggerRepository(),
      triggerConfiguration.triggerConfigurationReferencePersistence(),
      sqlConfiguration.ciIdService(),
      triggerService())
  }

}
