package com.xebialabs.xlrelease.triggers.event_based.validators

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.validation.{ExtendedValidationContext, ValidationContext, Validator}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.triggers.TriggerValidator
import com.xebialabs.xlrelease.triggers.event_based.EventBasedTrigger
import com.xebialabs.xlrelease.webhooks.mapping.MappedProperty.{PropertyValue, StringValue, VariableValue}
import com.xebialabs.xlrelease.webhooks.mapping.PropertyAddress

import scala.jdk.CollectionConverters._

class EventBasedTriggerValidator extends Validator[EventBasedTrigger] with TriggerValidator {

  override def validate(trigger: EventBasedTrigger, context: ValidationContext): Unit = {
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]
    val templateId = trigger.getTemplateId.orNull
    val releaseTitle = trigger.mappedProperties.asScala.find(_.targetProperty == "releaseTitle").collect {
      case x: StringValue => x.value
      case x: PropertyValue => x.sourceProperty
      case x: VariableValue => x.variableKey
    }.orNull

    checkTemplatePresence(extendedContext, trigger, templateId)
    checkTemplateFolderAndTriggerFolderMatch(extendedContext, trigger, templateId)
    checkTriggerFolderAndSourceFolderMatch()
    checkReleaseTitlePresence(extendedContext, trigger, releaseTitle)
    checkMalformedVariableNamesInMappings(trigger, extendedContext)
    checkInvalidMappedPropertyValues(trigger, extendedContext)
    checkFilterIsValid()

    def checkTriggerFolderAndSourceFolderMatch(): Unit = {
      if (trigger.eventSource != null) {
        val folderId: String = trigger.eventSource.getProperty("folderId")
        if (!isNullOrEmpty(trigger.getFolderId) && !isNullOrEmpty(folderId) && !trigger.getFolderId.contains(Ids.getName(folderId))) {
          extendedContext.error(trigger, "eventSource", s"Couldn't set eventSource that doesn't belong to the folder['${trigger.getFolderId}']")
        }
      }
    }

    def checkFilterIsValid(): Unit = {
      if (trigger.getEventFilter != null) {
        trigger.getEventFilter.validate(extendedContext)
      }
    }

  }

  private def checkInvalidMappedPropertyValues(trigger: EventBasedTrigger, extendedContext: ExtendedValidationContext): Unit = {
    val invalidPropertyValues = trigger.mappedProperties.asScala.collect {
      case propertyValue: PropertyValue if !PropertyAddress.hasProperty(Type.valueOf(trigger.eventType), PropertyAddress(propertyValue.sourceProperty)) =>
        propertyValue
    }
    if (invalidPropertyValues.nonEmpty) {
      extendedContext.focus(trigger, "mappedProperties").error(
        s"""|Mapped properties contain references to non-existing event properties:
            |${invalidPropertyValues.map(pv => s"'${trigger.eventType}' has no '${pv.sourceProperty}'").mkString(",\n")}""".stripMargin
      )
    }
  }
}
