package com.xebialabs.xlrelease.triggers.deployment_based.validators

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.plugin.api.validation.{ExtendedValidationContext, ValidationContext, Validator}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.triggers.TriggerValidator
import com.xebialabs.xlrelease.triggers.deployment_based.StatusWebhookTrigger

class StatusWebhookTriggerValidator extends Validator[StatusWebhookTrigger] with TriggerValidator {

  override def validate(trigger: StatusWebhookTrigger, context: ValidationContext): Unit = {
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]
    trigger.getEndpoint match {
      case null =>
        extendedContext.error(trigger, "endpoint", "Trigger endpoint is required")
      case _ if trigger.getEndpoint.getEventSource == null =>
        extendedContext.error(trigger, "endpoint.eventSource", "Trigger endpoint event source is required")
      case _ => // Valid endpoint and event source, no action needed
    }

    val templateId = trigger.getTemplateId.orNull
    val releaseTitle = trigger.getReleaseTitle.orNull
    val deploymentStatuses = trigger.getDeploymentStatuses
    val appTitle = trigger.getApplicationTitle

    checkTemplatePresence(extendedContext, trigger, templateId)
    checkTemplateFolderAndTriggerFolderMatch(extendedContext, trigger, templateId)
    checkTriggerFolderAndSourceFolderMatch(extendedContext, trigger)
    checkReleaseTitlePresence(extendedContext, trigger, releaseTitle)
    checkFilteringFieldsPresence(extendedContext, trigger, appTitle, deploymentStatuses)
    checkMalformedVariableNamesInMappings(trigger, extendedContext)
  }


  private def checkTriggerFolderAndSourceFolderMatch(extendedContext: ExtendedValidationContext, trigger: StatusWebhookTrigger): Unit =
    if (trigger.getEndpoint != null && trigger.getEndpoint.getEventSource != null) {
      val eventFolderId: String = trigger.getEndpoint.getEventSource.getProperty("folderId")
      val endpointFolderId: String = trigger.getEndpoint.getFolderId
      if (!isNullOrEmpty(endpointFolderId) && !isNullOrEmpty(eventFolderId) && !eventFolderId.equals(endpointFolderId)) {
        extendedContext.error(trigger, "eventSource", s"Event source folder id does not match endpoint folder id")
      }
      if (!isNullOrEmpty(trigger.getFolderId) && !isNullOrEmpty(endpointFolderId) && !isNullOrEmpty(eventFolderId) &&
        !trigger.getFolderId.contains(Ids.getName(eventFolderId))) {
        extendedContext.error(trigger, "eventSource", s"Couldn't set eventSource that doesn't belong to the folder['${trigger.getFolderId}']")
      }
    }

  private def checkFilteringFieldsPresence(extendedContext: ExtendedValidationContext, trigger: StatusWebhookTrigger,
                                   appTitle: String, deploymentStatuses: Set[String]): Unit = {
    if (isNullOrEmpty(appTitle)) {
      extendedContext.error(trigger, "applicationTitle", "Application title is required")
    }
    if (deploymentStatuses == null || deploymentStatuses.isEmpty) {
      extendedContext.error(trigger, "deploymentStatuses", s"Trigger does not have associated statuses")
    }
  }
}
