package com.xebialabs.xlrelease.triggers.scheduled

import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.repository.CiCloneHelper
import com.xebialabs.xlrelease.script._
import com.xebialabs.xlrelease.script.builder.{ScriptContextBuilder, _}
import com.xebialabs.xlrelease.script.jython.XlrJythonSupport
import com.xebialabs.xlrelease.webhooks.mapping.{VariableResolution, VariableResolver}
import grizzled.slf4j.Logging
import org.slf4j.MDC

import scala.jdk.CollectionConverters._
import scala.util.{Try, Using}

class TriggerScriptService(val scriptService: ScriptService, scriptVariables: ScriptVariables) extends Logging with XlrJythonSupport {

  import TriggerScriptService._

  def executeTrigger(originalReleaseTrigger: ReleaseTrigger): ReleaseTrigger = {
    try {
      val updatedTrigger = CiCloneHelper.cloneCi(originalReleaseTrigger)
      MDC.put(MDC_KEY_TRIGGER, updatedTrigger.getId)
      freezeVariables(updatedTrigger)
      Using.resource(triggerScriptContext(updatedTrigger)) { scriptContext =>
        executeScript(scriptContext)
        extractReleaseTriggerResults(scriptContext, updatedTrigger)
        updatedTrigger
      }
    } catch {
      case e: Throwable => throw new RuntimeException(s"Trigger ${originalReleaseTrigger.getId} execution failed", e)
    } finally {
      MDC.remove(MDC_KEY_TRIGGER)
    }
  }

  private def freezeVariables(trigger: ReleaseTrigger): ReleaseTrigger = {
    implicit val variableResolver: VariableResolver = new VariableResolver(trigger.getFolderId)(scriptVariables)

    val triggerType = trigger.getType
    val internalProperties = trigger.getInternalProperties.asScala ++ Seq("variables", "title")

    triggerType.getDescriptor.getPropertyDescriptors.asScala
      .filterNot(pd => pd.isHidden || internalProperties.contains(pd.getName))
      .foreach { pd =>
        Try {
          trigger.setProperty(pd.getName, VariableResolution.freezeVariables(trigger.getProperty(pd.getName)))
        }.recover {
          case ex => logger.warn(s"Unable to freeze variables in field ${pd.getName} of ${trigger.getId}: ${ex.getMessage}")
        }
      }
    trigger
  }

  private def extractReleaseTriggerResults(scriptContext: XlrScriptContext, trigger: ReleaseTrigger): Unit = {
    val triggerType = trigger.getType
    debug(s"Updating trigger with new state and/or variables on trigger : ${trigger.getId} $triggerType")
    trigger.getScriptVariableNames.asScala.foreach { propertyName =>
      val propertyValue = scriptContext.getAttribute(propertyName)
      trigger.setProperty(propertyName, propertyValue)
    }
    if (trigger.hasProperty("triggerState")) {
      val triggerState = scriptContext.getAttribute("triggerState").asInstanceOf[String]
      trigger.setTriggerState(triggerState)
    }
  }

}

object TriggerScriptService {
  final val MDC_KEY_TRIGGER = "trigger"

  def triggerScriptContext(releaseTrigger: ReleaseTrigger): XlrScriptContext = {
    new TriggerScriptContextBuilder(releaseTrigger).build()
  }

  private class TriggerScriptContextBuilder(releaseTrigger: ReleaseTrigger)
    extends ScriptContextBuilder {
    withCustomScriptApi().withLogger().withPythonSugar().withPythonGlobals().withPythonReleaseApi().withPythonUtilities()

    override protected def doBuild(context: XlrScriptContext): Unit = {
      EncryptionHelper.decrypt(releaseTrigger)
      context.addProperties(releaseTrigger, releaseTrigger.getType.getDescriptor.getPropertyDescriptors)

      val xlrScriptName = s"${releaseTrigger.getType.toString}[${releaseTrigger.getId}]"
      context.addScript(XlrScript.byContent(name = xlrScriptName, content = releaseTrigger.getScript, checkPermissions = false, wrap = false))
    }

  }

}
