package com.xebialabs.xlrelease.triggers.repository.persistence

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import com.xebialabs.xlrelease.api.v1.filter.TriggerFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.domain.status.TriggerExecutionStatus
import com.xebialabs.xlrelease.domain.{ReleaseTrigger, Trigger}
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository.Ids.{formatWithFolderId, getFolderlessId, getName}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceConstants.BLOB_TYPE
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{params, _}
import com.xebialabs.xlrelease.repository.sql.persistence.{FolderPersistence, PersistenceSupport}
import com.xebialabs.xlrelease.repository.sql.{SqlRepository, SqlRepositoryAdapter, persistence}
import com.xebialabs.xlrelease.repository.{Ids, RemoveByReleaseUid, RemoveByTemplateUid, RemoveTriggeredReleaseCondition}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.triggers.event_based.EventBasedTrigger
import com.xebialabs.xlrelease.utils.{CiHelper, FolderId}
import com.xebialabs.xlrelease.webhooks.mapping.MappedProperty.StringValue
import grizzled.slf4j.Logging
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource
import org.springframework.jdbc.core.support.SqlBinaryValue
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.util.StreamUtils.copyToString

import java.lang.String.format
import java.nio.charset.StandardCharsets
import java.sql.ResultSet
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@IsTransactional
class TriggerPersistence(val folderPersistence: FolderPersistence,
                         val sqlRepositoryAdapter: SqlRepositoryAdapter,
                         implicit val jdbcTemplate: JdbcTemplate,
                         implicit val dialect: Dialect)
  extends SqlRepository with PersistenceSupport with TriggerMapper with Logging {

  @IsReadOnly
  def numberOfTemplateTriggers(templateId: String): Int = {
    val QUERY_COUNT_TEMPLATE_TRIGGERS =
      s"""SELECT COUNT(1)
         | FROM
         | ${RELEASES.TABLE} r
         | JOIN ${TEMPLATE_TRIGGERS.TABLE} tt on tt.${TEMPLATE_TRIGGERS.RELEASE_UID} = r.${RELEASES.CI_UID}
         | JOIN ${TRIGGERS.TABLE} tr on tt.${TEMPLATE_TRIGGERS.TRIGGER_UID} = tr.${TRIGGERS.CI_UID}
         | WHERE r.${RELEASES.RELEASE_ID} = :${RELEASES.RELEASE_ID}
         |""".stripMargin
    namedTemplate.queryForObject(QUERY_COUNT_TEMPLATE_TRIGGERS, paramSource(RELEASES.RELEASE_ID -> getName(templateId)), classOf[Int])
  }


  private def triggerQueryBuilder = new TriggerQueryBuilder(folderPersistence, sqlRepositoryAdapter, dialect, namedTemplate)

  private val QUERY_TRIGGER_BY_TRIGGER_ID =
    s"""SELECT t.${TRIGGERS.ID},
       | t.${TRIGGERS.FOLDER_UID},
       | t.${TRIGGERS.CI_UID},
       | t.${TRIGGERS.ENABLED},
       | t.${TRIGGERS.CI_TYPE},
       | t.${TRIGGERS.TRIGGER_TITLE},
       | t.${TRIGGERS.DESCRIPTION},
       | t.${TRIGGERS.CONTENT},
       | t.${TRIGGERS.LAST_RUN_DATE},
       | t.${TRIGGERS.LAST_RUN_STATUS},
       | f.${FOLDERS.FOLDER_PATH},
       | f.${FOLDERS.FOLDER_ID}
       | FROM ${TRIGGERS.TABLE} t
       | INNER JOIN ${FOLDERS.TABLE} f on t.${TRIGGERS.FOLDER_UID} = f.${FOLDERS.CI_UID}
       | WHERE ${TRIGGERS.ID} = :${TRIGGERS.ID}
       |""".stripMargin

  def findById[T <: Trigger](triggerId: String): Option[T] = {
    try {
      Some(namedTemplate.queryForObject(
        QUERY_TRIGGER_BY_TRIGGER_ID,
        Map[String, Any](TRIGGERS.ID -> triggerId).asJava,
        triggerMapper(sqlRepositoryAdapter, folderPersistence)
      ))
    } catch {
      case _: EmptyResultDataAccessException => None
    }
  }

  def findBy(filter: TriggerFilters, pageable: Pageable): Page[Trigger] = {
    triggerQueryBuilder.from(filter).withPageable(pageable).build().execute()
  }

  private val STMT_INSERT_TRIGGER =
    s"""INSERT INTO ${TRIGGERS.TABLE}
       | ( ${TRIGGERS.ID}
       | , ${TRIGGERS.CI_UID}
       | , ${TRIGGERS.FOLDER_UID}
       | , ${TRIGGERS.ENABLED}
       | , ${TRIGGERS.TRIGGER_TITLE}
       | , ${TRIGGERS.CI_TYPE}
       | , ${TRIGGERS.DESCRIPTION}
       | , ${TRIGGERS.CONTENT}
       | , ${TRIGGERS.LAST_RUN_DATE}
       | , ${TRIGGERS.LAST_RUN_STATUS}
       | ) VALUES
       | ( :${TRIGGERS.ID}
       | , :${TRIGGERS.CI_UID}
       | , :${TRIGGERS.FOLDER_UID}
       | , :${TRIGGERS.ENABLED}
       | , :${TRIGGERS.TRIGGER_TITLE}
       | , :${TRIGGERS.CI_TYPE}
       | , :${TRIGGERS.DESCRIPTION}
       | , :${TRIGGERS.CONTENT}
       | , :${TRIGGERS.LAST_RUN_DATE}
       | , :${TRIGGERS.LAST_RUN_STATUS}
       | )
     """.stripMargin

  def insert(trigger: Trigger): Trigger = {
    val params: MapSqlParameterSource = upsertParams(trigger)
    sqlInsert(STMT_INSERT_TRIGGER, params)
    trigger
  }

  private val STMT_UPDATE_TRIGGER =
    s"""UPDATE ${TRIGGERS.TABLE}
       | SET ${TRIGGERS.FOLDER_UID} = :${TRIGGERS.FOLDER_UID},
       |  ${TRIGGERS.ENABLED} = :${TRIGGERS.ENABLED},
       |  ${TRIGGERS.TRIGGER_TITLE} = :${TRIGGERS.TRIGGER_TITLE},
       |  ${TRIGGERS.CI_TYPE} = :${TRIGGERS.CI_TYPE},
       |  ${TRIGGERS.DESCRIPTION} = :${TRIGGERS.DESCRIPTION},
       |  ${TRIGGERS.CONTENT} = :${TRIGGERS.CONTENT},
       |  ${TRIGGERS.LAST_RUN_DATE} = :${TRIGGERS.LAST_RUN_DATE},
       |  ${TRIGGERS.LAST_RUN_STATUS} = :${TRIGGERS.LAST_RUN_STATUS}
       | WHERE ${TRIGGERS.ID} = :${TRIGGERS.ID}
     """.stripMargin

  //TODO Does thid work??
  def update(trigger: Trigger): Unit = {
    val params: MapSqlParameterSource = upsertParams(trigger)
    sqlUpdate(STMT_UPDATE_TRIGGER, params, checkCiUpdated(trigger.getId))
  }

  private def upsertParams(trigger: Trigger): MapSqlParameterSource = {
    val params = new MapSqlParameterSource()
    val folderUid = folderPersistence.getUid(trigger.getFolderId)
    params.addValue(TRIGGERS.CI_UID, trigger.getCiUid)
    params.addValue(TRIGGERS.FOLDER_UID, folderUid)
    params.addValue(TRIGGERS.ENABLED, trigger.isEnabled.asInteger)
    params.addValue(TRIGGERS.TRIGGER_TITLE, trigger.getTitle.trimAndTruncate(persistence.LONG_VARCHAR_LENGTH))
    params.addValue(TRIGGERS.CI_TYPE, trigger.getType.toString)
    params.addValue(TRIGGERS.DESCRIPTION, trigger.getDescription.trimAndTruncate(persistence.LONG_VARCHAR_LENGTH))
    params.addValue(TRIGGERS.CONTENT, new SqlBinaryValue(serialize(trigger).getBytes(StandardCharsets.UTF_8)), BLOB_TYPE)
    params.addValue(TRIGGERS.LAST_RUN_DATE, trigger.getLastRunDate)
    params.addValue(TRIGGERS.LAST_RUN_STATUS, Option(trigger.getLastRunStatus).map(_.value()).orNull)
    if (Ids.isInRelease(trigger.getId)) {
      params.addValue(TRIGGERS.ID, getFolderlessId(trigger.getId.normalized))
    } else {
      val triggerId = Ids.getName(getFolderlessId(trigger.getId.normalized))
      trigger.setId(formatWithFolderId(trigger.getFolderId, triggerId))
      params.addValue(TRIGGERS.ID, triggerId)
    }
    params
  }

  private def serialize(trigger: Trigger): String = {
    val triggerCopy = cloneCi(trigger)
    triggerCopy.setId(getFolderlessId(trigger.getId))
    triggerCopy.setFolderId(null)
    CiSerializerHelper.serialize(triggerCopy)
  }

  private val STMT_DELETE_TRIGGER =
    s"""DELETE FROM ${TRIGGERS.TABLE}
       | WHERE ${TRIGGERS.ID} = :${TRIGGERS.ID}
     """.stripMargin

  def delete(triggerId: String): Unit = {
    sqlUpdate(STMT_DELETE_TRIGGER, params(
      TRIGGERS.ID -> getFolderlessId(triggerId)
    ), checkCiUpdated(triggerId))
  }

  private val STMT_GET_ENABLED_TRIGGER_IDS_FOR_TEMPLATE_ID =
    s"""SELECT ${TRIGGERS.ID}
       | FROM ${TRIGGERS.TABLE} t
       | JOIN ${TEMPLATE_TRIGGERS.TABLE} tt ON tt.${TEMPLATE_TRIGGERS.TRIGGER_UID} = t.${TRIGGERS.CI_UID}
       | JOIN ${RELEASES.TABLE} r ON tt.${TEMPLATE_TRIGGERS.RELEASE_UID} = r.${RELEASES.CI_UID}
       |WHERE
       | r.${RELEASES.RELEASE_ID} = :templateId
       | AND t.${TRIGGERS.ENABLED} = 1
     """.stripMargin

  def getEnabledReleaseTriggerIds(templateId: String): Seq[String] = {
    sqlQuery(STMT_GET_ENABLED_TRIGGER_IDS_FOR_TEMPLATE_ID, params(
      "templateId" -> getName(templateId)
    ), triggerIdRowMapper).toSeq
  }

  private val STMT_GET_TEMPLATE_TRIGGER_ROW_BY_RELEASE =
    s"""SELECT ${TEMPLATE_TRIGGERS.RELEASE_UID}, ${TEMPLATE_TRIGGERS.TRIGGER_UID}
       | FROM ${TEMPLATE_TRIGGERS.TABLE}
       | WHERE ${TEMPLATE_TRIGGERS.RELEASE_UID} = :releaseCiUid
       |""".stripMargin

  private val templateTriggerRowMapper: RowMapper[TemplateTriggerRow] = (rs: ResultSet, _: Int) => {
    TemplateTriggerRow(rs.getString(TEMPLATE_TRIGGERS.RELEASE_UID), rs.getString(TEMPLATE_TRIGGERS.TRIGGER_UID))
  }

  def getTemplateTriggerRows(releaseCiUid: CiUid): Seq[TemplateTriggerRow] = {
    sqlQuery(STMT_GET_TEMPLATE_TRIGGER_ROW_BY_RELEASE,
      params("releaseCiUid" -> releaseCiUid),
      templateTriggerRowMapper
    ).toSeq
  }

  private val INSERT_TEMPLATE_TRIGGER_ROW =
    s"""INSERT INTO ${TEMPLATE_TRIGGERS.TABLE}
       | (${TEMPLATE_TRIGGERS.RELEASE_UID}, ${TEMPLATE_TRIGGERS.TRIGGER_UID})
       | VALUES (:releaseCiUid, :triggerCiUid)
       |""".stripMargin

  def insertTemplateTriggerRow(row: TemplateTriggerRow): Unit = {
    sqlInsert(INSERT_TEMPLATE_TRIGGER_ROW,
      params(
        "releaseCiUid" -> row.releaseCiUid,
        "triggerCiUid" -> row.triggerCiUid,
      ))
  }

  private val STMT_GET_TRIGGERED_RELEASES_COUNT =
    s"SELECT COUNT(1) FROM ${TRIGGERED_RELEASES.TABLE} WHERE ${TRIGGERED_RELEASES.TEMPLATE_UID}=:${TRIGGERED_RELEASES.TEMPLATE_UID}"

  def getRunningTriggeredReleasesCount(templateUid: CiUid): Int = {
    namedTemplate.queryForObject(STMT_GET_TRIGGERED_RELEASES_COUNT, paramSource(TRIGGERED_RELEASES.TEMPLATE_UID -> templateUid), classOf[Int])
  }

  private val INSERT_TRIGGERED_RELEASE_ROW =
    s"""INSERT INTO ${TRIGGERED_RELEASES.TABLE}
       | (${TRIGGERED_RELEASES.TEMPLATE_UID}, ${TRIGGERED_RELEASES.TRIGGER_UID}, ${TRIGGERED_RELEASES.RELEASE_UID})
       | VALUES (:${TRIGGERED_RELEASES.TEMPLATE_UID}, :${TRIGGERED_RELEASES.TRIGGER_UID}, :${TRIGGERED_RELEASES.RELEASE_UID})
       |""".stripMargin

  def insertTriggeredReleaseRow(row: TriggeredReleaseRow): Unit = {
    sqlInsert(INSERT_TRIGGERED_RELEASE_ROW,
      params(
        TRIGGERED_RELEASES.TEMPLATE_UID -> row.templateUid,
        TRIGGERED_RELEASES.TRIGGER_UID -> row.triggerUid,
        TRIGGERED_RELEASES.RELEASE_UID -> row.releaseUid
      ))
  }

  private val STMT_DELETE_TRIGGERED_RELEASE_BY_RELEASE_UID =
    s"""DELETE FROM ${TRIGGERED_RELEASES.TABLE}
       | WHERE ${TRIGGERED_RELEASES.RELEASE_UID} = :${TRIGGERED_RELEASES.RELEASE_UID}
       |""".stripMargin

  private val STMT_DELETE_TRIGGERED_RELEASE_BY_TEMPLATE_UID =
    s"""DELETE FROM ${TRIGGERED_RELEASES.TABLE}
       | WHERE ${TRIGGERED_RELEASES.TEMPLATE_UID} = :${TRIGGERED_RELEASES.TEMPLATE_UID}
       |""".stripMargin

  def removeTriggeredReleasesRow(condition: RemoveTriggeredReleaseCondition): Unit = {
    condition match {
      case RemoveByReleaseUid(releaseUid) =>
        sqlUpdate(
          STMT_DELETE_TRIGGERED_RELEASE_BY_RELEASE_UID,
          params(TRIGGERED_RELEASES.RELEASE_UID -> releaseUid),
          checkDeleteResult(s"No rows were deleted for releaseUid[$releaseUid]")
        )
      case RemoveByTemplateUid(templateUid) =>
        sqlUpdate(
          STMT_DELETE_TRIGGERED_RELEASE_BY_TEMPLATE_UID,
          params(TRIGGERED_RELEASES.TEMPLATE_UID -> templateUid),
          checkDeleteResult(s"No rows were deleted for templateUid[$templateUid]")
        )
    }
  }

  private def checkDeleteResult(msg: String): Int => Unit = {
    case 0 => logger.debug(msg)
    case _ => ()
  }
}

case class TemplateTriggerRow(releaseCiUid: CiUid, triggerCiUid: CiUid)

case class TriggeredReleaseRow(templateUid: CiUid, triggerUid: CiUid, releaseUid: CiUid)

trait TriggerMapper {
  // scalastyle:off cyclomatic.complexity
  // scalastyle:off method.length
  def triggerMapper[T <: Trigger]: (SqlRepositoryAdapter, FolderPersistence) => RowMapper[T] = (sqlRepositoryAdapter, folderPersistence) => (rs, _) => {
    def getFullTemplateId(templateId: String): String = {
      if (null != templateId) {
        val partialId = Ids.getName(templateId)
        val maybeTemplateFolderId = sqlRepositoryAdapter.releasePersistence.findFolderIdByReleaseId(partialId)
        maybeTemplateFolderId match {
          case Some(folderId) => Ids.formatWithFolderId(folderId, partialId)
          case None => templateId
        }
      } else {
        templateId
      }
    }

    def getFullFolderId(folderId: String): String = {
      if (null != folderId) {
        val partialId = Ids.getName(folderId)
        folderPersistence.findById(partialId, 0).toFolder.getId
      } else {
        folderId
      }
    }

    val contentStream = rs.getBinaryStream(TRIGGERS.CONTENT)
    val trigger: T = if (null != contentStream) {
      using(contentStream) { is =>
        Try(copyToString(is, StandardCharsets.UTF_8)) match {
          case Success(json) => CiSerializerHelper.deserialize(json, sqlRepositoryAdapter).asInstanceOf[T]
          case Failure(ex) => throw new IllegalStateException("Failed to load trigger", ex)
        }
      }
    } else {
      val triggerType = Type.valueOf(rs.getString(TRIGGERS.CI_TYPE))
      triggerType.getDescriptor.newInstance[T](null)
    }
    trigger.setCiUid(rs.getString(TRIGGERS.CI_UID))
    trigger.setTitle(rs.getString(TRIGGERS.TRIGGER_TITLE))
    trigger.setDescription(rs.getString(TRIGGERS.DESCRIPTION))
    trigger.setEnabled(rs.getInt(TRIGGERS.ENABLED).asBoolean)
    val folderId = FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)
    trigger.setFolderId(folderId.absolute)
    trigger.setLastRunDate(rs.getTimestamp(TRIGGERS.LAST_RUN_DATE))
    trigger.setLastRunStatus(Option(rs.getString(TRIGGERS.LAST_RUN_STATUS)).map(value => TriggerExecutionStatus.valueOf(value.toUpperCase)).orNull)
    val triggerId = formatWithFolderId(trigger.getFolderId, rs.getString(TRIGGERS.ID))
    rewriteWithNewTriggerId(trigger, triggerId)

    // regenerate full templateId and folderId for triggers that have been loaded from the database as folder might have been moved
    trigger match {
      case t: ReleaseTrigger =>
        t.setTemplate(getFullTemplateId(t.getTemplate))
        t.setReleaseFolder(getFullFolderId(t.getReleaseFolder))
      case t: EventBasedTrigger =>
        t.mappedProperties.asScala.find(_.targetProperty == "templateId").foreach {
          case s: StringValue =>
            s.setValue(getFullTemplateId(s.value))
          case _ => () // do nothing
        }
        t.mappedProperties.asScala.find(_.targetProperty == "releaseFolderId").foreach {
          case s: StringValue =>
            s.setValue(getFullFolderId(s.value))
          case _ => () // do nothing
        }
      case _ => () // do nothing
    }

    trigger
  }

  def triggerIdRowMapper: RowMapper[String] = (rs, _) => {
    rs.getString(TRIGGERS.ID)
  }

  private def rewriteWithNewTriggerId(trigger: Trigger, newId: String): Unit = {
    val oldId = trigger.getId
    val oldIdPattern = format(".*%s(?=/|$)", oldId) // match old CI ID ending with a "/" or end of line

    CiHelper.getNestedCis(trigger).asScala.foreach(nestedCi => {
      if (nestedCi.getId != null && nestedCi.getId.contains(oldId)) {
        val rewrittenId = nestedCi.getId.replaceFirst(oldIdPattern, newId)
        nestedCi.setId(rewrittenId)
      }
    })

    CiHelper.getExternalReferences(trigger).asScala.foreach(referencedCi => {
      if (referencedCi.getId != null && referencedCi.getId.contains(oldId)) {
        val rewrittenId = referencedCi.getId.replaceFirst(oldIdPattern, newId)
        referencedCi.setId(rewrittenId)
      }
    })
  }
}
