package com.xebialabs.xlrelease.triggers.repository.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.domain.{Release, ReleaseTrigger, Trigger}
import com.xebialabs.xlrelease.repository.sql.ConfigurationReferencesSupport
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationReferencePersistence
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, ReleasePersistence}
import com.xebialabs.xlrelease.repository.{Ids, PersistenceInterceptor}
import com.xebialabs.xlrelease.triggers.event_based.EventBasedTrigger
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.JdbcTemplate

@IsTransactional
class TriggerPersistenceInterceptor(val triggerPersistence: TriggerPersistence,
                                    val releasePersistence: ReleasePersistence,
                                    val configurationPersistence: ConfigurationReferencePersistence,
                                    val jdbcTemplate: JdbcTemplate,
                                    val dialect: Dialect)
  extends PersistenceInterceptor[Trigger] with Logging with ReleaseTriggerPersistence with ConfigurationReferencesSupport {

  override def getReleasePersistence: ReleasePersistence = releasePersistence

  override def afterCreate(ci: Trigger): Unit = {
    logger.info(s"Trigger created: '${ci.getType}' ciUid:'${ci.getCiUid}' id: '${ci.getId}'")
    updateConfigurationRefs(ci)
    ci match {
      case rt: ReleaseTrigger => link(rt)
      case rt: EventBasedTrigger => link(rt)
      case _ => // nothing
    }
  }

  override def afterUpdate(ci: Trigger): Unit = {
    updateConfigurationRefs(ci)
    ci match {
      case rt: ReleaseTrigger => updateLink(rt)
      case rt: EventBasedTrigger => updateLink(rt)
      case _ => // nothing
    }
  }

  override def onDelete(triggerId: String): Unit = {
    val trigger = triggerPersistence.findById[Trigger](Ids.getFolderlessId(triggerId.normalized))
    trigger match {
      case Some(rt: ReleaseTrigger) => unlink(rt.getCiUid)
      case Some(rt: EventBasedTrigger) => unlink(rt.getCiUid)
      case _ => // nothing
    }
  }

  private def unlink(triggerCiUid: CiUid) = {
    sqlUpdate(
      s"DELETE FROM ${TEMPLATE_TRIGGERS.TABLE} WHERE ${TEMPLATE_TRIGGERS.TRIGGER_UID} = :${TEMPLATE_TRIGGERS.TRIGGER_UID}",
      paramSource(TEMPLATE_TRIGGERS.TRIGGER_UID -> triggerCiUid),
      numOfRows => logger.info(s"DELETED ${numOfRows} from ${TEMPLATE_TRIGGERS.TABLE}")
    )

  }

}

/**
 * Intercepts Template CRUD operations.
 * Main purpose is to handle Trigger lifecycle.
 *
 * @param jdbcTemplate
 * @param dialect
 */
@IsTransactional
class TriggerTemplatePersistenceInterceptor(val releasePersistence: ReleasePersistence,
                                            val triggerPersistence: TriggerPersistence,
                                            val jdbcTemplate: JdbcTemplate,
                                            val dialect: Dialect)
  extends PersistenceInterceptor[Release] with Logging with ReleaseTriggerPersistence {

  val NL = System.lineSeparator()

  override def getReleasePersistence: ReleasePersistence = releasePersistence

  override def onDelete(templateId: String): Unit = {
    unlink(templateId)
  }

  private[repository] def unlink(templateId: String): Unit = {
    val releaseCiUid: CiUid = releasePersistence.findUidByReleaseId(templateId).get
    sqlUpdate(
      s"DELETE FROM ${TEMPLATE_TRIGGERS.TABLE} WHERE ${TEMPLATE_TRIGGERS.RELEASE_UID} = :${TEMPLATE_TRIGGERS.RELEASE_UID}",
      paramSource(TEMPLATE_TRIGGERS.RELEASE_UID -> releaseCiUid),
      _ => ()
    )
  }

}


trait ReleaseTriggerPersistence extends PersistenceSupport with Logging {
  def getReleasePersistence: ReleasePersistence

  def releasePersistence: ReleasePersistence

  private[repository] def link(releaseTrigger: ReleaseTrigger): Unit = {
    val releaseCiUid = releasePersistence.findUidByReleaseId(releaseTrigger.getTemplate).get
    sqlInsert(s"INSERT INTO ${TEMPLATE_TRIGGERS.TABLE} VALUES(:${TEMPLATE_TRIGGERS.RELEASE_UID}, :${TEMPLATE_TRIGGERS.TRIGGER_UID})",
      paramSource(
        TEMPLATE_TRIGGERS.RELEASE_UID -> releaseCiUid,
        TEMPLATE_TRIGGERS.TRIGGER_UID -> releaseTrigger.getCiUid
      )
    )
  }

  private[repository] def link(eventBasedTrigger: EventBasedTrigger): Unit = {
    eventBasedTrigger.getTemplateId.flatMap(getReleasePersistence.findUidByReleaseId).foreach(templateCiUid => {
      sqlInsert(s"INSERT INTO ${TEMPLATE_TRIGGERS.TABLE} VALUES(:${TEMPLATE_TRIGGERS.RELEASE_UID}, :${TEMPLATE_TRIGGERS.TRIGGER_UID})",
        paramSource(
          TEMPLATE_TRIGGERS.RELEASE_UID -> templateCiUid,
          TEMPLATE_TRIGGERS.TRIGGER_UID -> eventBasedTrigger.getCiUid
        )
      )
    })
  }

  private[repository] def updateLink(releaseTrigger: ReleaseTrigger): Unit = {
    releasePersistence.findUidByReleaseId(releaseTrigger.getTemplate).foreach { releaseCiUid =>
      val STMT_UPDATE_TEMPLATE_TRIGGERS =
        s"""UPDATE ${TEMPLATE_TRIGGERS.TABLE}
           | SET ${TEMPLATE_TRIGGERS.RELEASE_UID} = :${TEMPLATE_TRIGGERS.RELEASE_UID}
           | WHERE ${TEMPLATE_TRIGGERS.TRIGGER_UID} = :${TEMPLATE_TRIGGERS.TRIGGER_UID}
           |""".stripMargin
      sqlUpdate(STMT_UPDATE_TEMPLATE_TRIGGERS, paramSource(
        TEMPLATE_TRIGGERS.RELEASE_UID -> releaseCiUid,
        TEMPLATE_TRIGGERS.TRIGGER_UID -> releaseTrigger.getCiUid
      ), _ => ())
    }
  }

  private[repository] def updateLink(eventBasedTrigger: EventBasedTrigger): Unit = {
    eventBasedTrigger.getTemplateId.flatMap(getReleasePersistence.findUidByReleaseId).foreach(templateCiUid => {
      val STMT_UPDATE_TEMPLATE_TRIGGERS =
        s"""UPDATE ${TEMPLATE_TRIGGERS.TABLE}
           | SET ${TEMPLATE_TRIGGERS.RELEASE_UID} = :${TEMPLATE_TRIGGERS.RELEASE_UID}
           | WHERE ${TEMPLATE_TRIGGERS.TRIGGER_UID} = :${TEMPLATE_TRIGGERS.TRIGGER_UID}
           |""".stripMargin
      sqlUpdate(STMT_UPDATE_TEMPLATE_TRIGGERS, paramSource(
        TEMPLATE_TRIGGERS.RELEASE_UID -> templateCiUid,
        TEMPLATE_TRIGGERS.TRIGGER_UID -> eventBasedTrigger.getCiUid
      ), _ => ())
    })
  }
}
