package com.xebialabs.xlrelease.upgrade

import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.domain.ScheduledTrigger
import com.xebialabs.xlrelease.quartz.release.scheduler.ReleaseSchedulerService
import com.xebialabs.xlrelease.repository.TriggerRepository
import com.xebialabs.xlrelease.triggers.scheduled.ScheduledJobService
import com.xebialabs.xlrelease.triggers.scheduled.quartz.QuartzScheduledJobService
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import grizzled.slf4j.Logging
import org.springframework.data.domain.Pageable
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

/**
 * As of 22.3.x, System was executing all release scheduled triggers and quartz triggers on the same scheduler.
 * From 23.1, release scheduled triggers and quartz triggers are executed on a dedicated scheduler.
 * System has to upgrade scheduler for existing release triggers in order to avoid duplicate triggers in quartz table.
 */
@Component
class XLRelease231TriggerSchedulerUpgrade(triggerRepository: TriggerRepository,
                                          releaseSchedulerService: ReleaseSchedulerService,
                                          scheduledJobService: ScheduledJobService)
  extends Upgrade with Logging {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "23.1.0#3")

  override def doUpgrade(): Boolean = {
    logger.info(s"Changing scheduler for triggers")
    val oldQuartzScheduledJobService = new QuartzScheduledJobService(releaseSchedulerService)
    val scheduledTriggerTypes = DescriptorRegistry.getSubtypes(Type.valueOf(classOf[ScheduledTrigger])).asScala.toList
    val triggersPage = triggerRepository.findByType(scheduledTriggerTypes, Pageable.unpaged())

    if (triggersPage.hasContent) {
      val triggers = triggersPage.getContent.asScala
      triggers.foreach { trigger =>
        val scheduledTrigger = trigger.asInstanceOf[ScheduledTrigger]
        try {
          // un-schedule a trigger from old scheduler
          oldQuartzScheduledJobService.unschedule(scheduledTrigger)
        } catch {
          case ex: Exception => logger.error(s"Unable to un-schedule trigger [${scheduledTrigger.getId}]", ex)
        }
        if (scheduledTrigger.isEnabled) {
          try {
            // schedule a trigger on a new dedicated trigger scheduler
            scheduledJobService.schedule(scheduledTrigger)
          } catch {
            case ex: Exception => logger.error(s"Unable to schedule trigger [${scheduledTrigger.getId}]", ex)
          }
        }
      }
    }
    logger.debug(s"Scheduler change for triggers completed")
    true
  }


}
