package com.xebialabs.xlrelease.triggers.service.impl

import com.xebialabs.deployit.plugin.api.validation.ValidationMessage
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.triggers.action.TriggerAction
import org.springframework.core.Ordered

import java.util

/**
  * This is the main triggerLifecycle factory
  */
class DefaultTriggerLifecycle(implementations: List[TriggerLifecycle[_]]) extends TriggerLifecycle[Trigger] {

  override def execute(trigger: Trigger, executionContext: TriggerExecutionContext): TriggerExecutionResult[Trigger] = {
    findImplementation(trigger).execute(trigger, executionContext)
  }

  override def enable(trigger: Trigger, checkReferencePermissions: Boolean): Unit = {
    val impl = findImplementation(trigger)
    impl.validate(trigger, checkReferencePermissions) // safety net
    impl.enable(trigger, checkReferencePermissions)
  }

  override def disable(trigger: Trigger): Unit = findImplementation(trigger).disable(trigger)

  override def validate(trigger: Trigger, checkReferencePermissions: Boolean = true): util.List[ValidationMessage] = {
    findImplementation(trigger).validate(trigger, checkReferencePermissions)
  }

  override def supports(trigger: Trigger): Boolean = true

  override def getOrder: Int = Ordered.HIGHEST_PRECEDENCE

  private def findImplementation(trigger: Trigger): TriggerLifecycle[Trigger] = {
    implementations
      .sortBy(_.getOrder)
      .find(t => t.supports(trigger))
      .getOrElse(throw new IllegalArgumentException(s"Unsupported trigger type '${trigger.getType}''"))
      .asInstanceOf[TriggerLifecycle[Trigger]]
  }

  override def getTriggerAction(trigger: Trigger, executionContext: TriggerExecutionContext): TriggerAction =
    findImplementation(trigger).getTriggerAction(trigger, executionContext)

}
