package com.xebialabs.xlrelease.triggers.service.impl

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.actors.utils.TriggerActorLifecycleUtils
import com.xebialabs.xlrelease.api.v1.filter.TriggerFilters
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.repository.TriggerRepository
import com.xebialabs.xlrelease.service.{CiIdService, FeatureService}
import com.xebialabs.xlrelease.support.pekko.spring.ScalaSpringSupport
import com.xebialabs.xlrelease.triggers.actors.TriggerActor._
import com.xebialabs.xlrelease.triggers.actors.TriggerActorHolder
import com.xebialabs.xlrelease.triggers.management.repository.TriggerManagementRepository
import com.xebialabs.xlrelease.triggers.service.TriggerService
import com.xebialabs.xlrelease.utils.FolderId
import com.xebialabs.xlrelease.view.ReleaseTriggerOverview
import grizzled.slf4j.Logging
import org.apache.pekko.actor.ActorRef
import org.apache.pekko.pattern.ask
import org.apache.pekko.util.Timeout
import org.springframework.context.{ApplicationContext, ApplicationContextAware}
import org.springframework.data.domain.{Page, Pageable}

import java.util.{List => JList}
import scala.beans.BeanProperty
import scala.concurrent.{Await, TimeoutException}
import scala.jdk.CollectionConverters._

class DefaultTriggerService(triggerActorHolder: TriggerActorHolder,
                            xlrConfig: XlrConfig,
                            ciIdService: CiIdService,
                            triggerRepository: TriggerRepository,
                            triggerManagementRepository: TriggerManagementRepository)
  extends TriggerService with Logging with TriggerFeatureService {

  lazy val actorRef: ActorRef = triggerActorHolder.actorRef()

  override def findBy(triggerFilters: TriggerFilters, pageable: Pageable): Page[ReleaseTriggerOverview] = {
    triggerManagementRepository.findBy(triggerFilters, pageable)
  }

  override def findById(triggerId: String): Trigger = triggerRepository.find[Trigger](triggerId)

  // TODO(triggers-tech-debt): Add the query to fetch folderId only.
  override def getFolderId(triggerId: String) = findById(triggerId).getFolderId

  override def enableTriggers(triggerIds: JList[String]): JList[String] = {
    triggerIds.asScala.foreach(updateTriggerStatus(_, true, true))
    triggerIds
  }

  override def disableTriggers(triggerIds: JList[String]): JList[String] = {
    triggerIds.asScala.foreach(updateTriggerStatus(_, false, true))
    triggerIds
  }

  override def enableAllAccessibleTriggers(): JList[String] = {
    enableTriggers(findAllEditableTriggerIds)
  }

  override def disableAllAccessibleTriggers(): JList[String] = {
    disableTriggers(findAllEditableTriggerIds)
  }


  override def addTrigger(trigger: Trigger): Trigger = askAndAwait {
    trigger.setFolderId(FolderId(trigger.getFolderId).absolute)
    val triggerId = ciIdService.getUniqueId(Type.valueOf(classOf[Trigger]), trigger.getFolderId)
    trigger.setId(triggerId)
    AddTrigger(trigger)
  }

  override def updateVariables(trigger: Trigger): Unit = askAndAwait {
    UpdateVariables(trigger)
  }

  override def updateTriggerStatus(triggerId: String, enabled: Boolean, checkReferencePermissions: Boolean): Unit = askAndAwait {
    UpdateTriggerStatus(triggerId, enabled, checkReferencePermissions)
  }

  override def disableTriggerAuto(triggerId: String): Unit = askAndAwait {
    AutoDisableTrigger(triggerId)
  }

  override def updateInternalState(trigger: Trigger, updateProperties: Seq[String] = Seq.empty): Unit = askAndAwait {
    UpdateInternalState(trigger, updateProperties)
  }

  override def updateTrigger(trigger: Trigger): Trigger = askAndAwait {
    UpdateTrigger(trigger)
  }

  override def refreshTrigger(trigger: Trigger): Unit = tell {
    RefreshTrigger(trigger)
  }

  override def deleteTrigger(triggerId: String): Unit = askAndAwait {
    DeleteTrigger(triggerId)
  }

  override def runNowTrigger(triggerId: String): Unit = this.executeSync(triggerId, RunNowExecutionContext())

  override def execute(triggerId: String, executionContext: TriggerExecutionContext): Unit = tell {
    ExecuteTrigger(triggerId, executionContext)
  }

  override def executeSync(triggerId: String, executionContext: TriggerExecutionContext): Trigger = {
    val result: ExecuteTriggerResult = askAndAwait {
      ExecuteTrigger(triggerId, executionContext)
    }

    result.result match {
      case Left(t: Trigger) => t
      case Right(message) => throw new RuntimeException(message)
    }
  }

  private def findAllEditableTriggerIds(): java.util.List[String] = {
    triggerManagementRepository.findAllEditableTriggerIds()
  }

  private def tell(msg: AnyRef): Unit = {
    if (isEnabled) {
      logger.debug(s"Forwarding release action '$msg' to '$actorRef'")
      actorRef ! msg
    } else {
      logger.warn(s"Ignoring received '$msg' while service is disabled")
    }
  }

  private def askAndAwait[T](msg: AnyRef): T = {
    if (isEnabled) {
      implicit val askTimeout: Timeout = xlrConfig.timeouts.releaseActionResponse
      try {
        Await.result(
          actorRef ? msg,
          askTimeout.duration
        ).asInstanceOf[T]
      } catch {
        case e: TimeoutException => throw new RuntimeException(e)
        case e: InterruptedException => throw new RuntimeException(e)
      }
    } else {
      throw new IllegalStateException(s"Ignoring received '$msg' while service is disabled")
    }
  }
}

trait TriggerFeatureService extends FeatureService with ApplicationContextAware with ScalaSpringSupport with Logging {
  self: TriggerService =>
  @volatile
  private var enabled: Boolean = true
  @BeanProperty
  var applicationContext: ApplicationContext = _
  private lazy val triggerActorLifecycleUtils = springBean[TriggerActorLifecycleUtils]

  override def name(): String = "TriggerService"

  override def enable(): Unit = {
    this.enabled = true
  }

  override def disable(): Unit = {
    try {
      this.disableAllAccessibleTriggers()
      this.enabled = false
      this.triggerActorLifecycleUtils.terminateAllTriggerActors()
    } finally {
      this.enabled = false
    }
  }

  override def stop(): Unit = {
    this.enabled = false
  }

  override def isEnabled: Boolean = enabled
}
