import { call, put } from 'redux-saga/effects';
import isUndefined from 'lodash/isUndefined';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { ILocationService } from 'angular';
import { registerSidebarProvider, SidebarProvider } from '@xlr-ui/app/features/custom-ui/sidebar.registry';
import { getFolderOrWaitForIt } from '@xlr-ui/app/features/folders/ducks/folders.saga';
import { navigation } from '@xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { Folder } from '@xlr-ui/app/types';

const { setSidebarMenu, setBackItem } = navigation.actions;

interface TriggerParams {
    folderId?: string;
}

export class TriggerSidebarProvider implements SidebarProvider {
    isSupported(path: string): boolean {
        return path.startsWith('/trigger-management/Trigger');
    }

    *provide(path: string) {
        const $location: ILocationService = yield call(getAngularService, '$location');
        const params: TriggerParams = $location.search();

        yield put(
            setSidebarMenu({
                withFolder: false,
                sidebarMenu: [
                    {
                        text: 'Properties',
                        pathSuffix: yield call(this.getPath, path, '', params),
                        selected: !path.endsWith('/logs'),
                        startIconId: 'configuration-preferences',
                    },
                    {
                        text: 'Activity Logs',
                        pathSuffix: yield call(this.getPath, path, '/logs', params),
                        selected: path.endsWith('/logs'),
                        startIconId: 'progress',
                    },
                ],
            }),
        );
        if (isUndefined(params.folderId) || params.folderId === 'Applications') {
            // LEGACY: for old triggers that are attached to root templates
            // this should go to the template trigger but we don't have access to it
            yield put(
                setBackItem({
                    pathSuffix: `templates`,
                    title: 'templates',
                    text: 'Templates',
                }),
            );
        } else {
            const selectedFolder: Folder = yield call(getFolderOrWaitForIt);
            yield put(
                setBackItem({
                    pathSuffix: `folders/${selectedFolder.id}/triggers`,
                    title: 'triggers',
                    text: '', // we don't have here the title of the trigger, this will be filled by setBreadcrumbsSaga
                }),
            );
        }
    }

    getPath(path: string, subpage: string, params: TriggerParams) {
        // trigger should be in a reducer and don't make this mess, at some point in future i guess
        const fixedPath = path.replace('/trigger-management/', 'trigger-management/').replace('/logs', '');
        return params.folderId ? `${fixedPath}${subpage}?folderId=${params.folderId}` : `${fixedPath}${subpage}`;
    }
}

registerSidebarProvider(new TriggerSidebarProvider());
