package com.xebialabs.xlrelease.triggers.event_based

import com.google.common.base.Strings.isNullOrEmpty
import com.jayway.jsonpath.spi.json.{JacksonJsonProvider, JsonProvider}
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.{Metadata, Property}
import com.xebialabs.deployit.plugin.api.validation.ExtendedValidationContext
import com.xebialabs.deployit.plumbing.ExecutionOutputWriter
import com.xebialabs.platform.script.jython.{JythonSupport, ScriptSource}
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlrelease.repository.Ids.findFolderId
import com.xebialabs.xlrelease.script.builder.ScriptContextBuilder
import com.xebialabs.xlrelease.script.groovy.GroovyScriptExecutor
import com.xebialabs.xlrelease.script.jython.JythonScriptExecutor
import com.xebialabs.xlrelease.script.{OutputHandler, ScriptExecutor, XlrScript, XlrScriptContext}
import com.xebialabs.xlrelease.triggers.event_based.EventFilter.FIELD_MAX_LENGTH
import com.xebialabs.xlrelease.triggers.event_based.EventFilterOperator._
import com.xebialabs.xlrelease.triggers.event_based.EventFilterValueType._
import com.xebialabs.xlrelease.triggers.event_based.ExpressionEventFilter.ALLOWED_PREFIXES
import com.xebialabs.xlrelease.triggers.event_based.JythonEventFilter.libraries
import com.xebialabs.xlrelease.triggers.event_based.ScriptBasedEventFilter.RESULT_ATTRIBUTE
import com.xebialabs.xlrelease.utils.{DateVariableUtils, SensitiveValueScrubber}
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.webhooks.mapping.PropertyAddress.any2AnyRef
import com.xebialabs.xlrelease.webhooks.mapping.VariableResolution
import com.xebialabs.xlrelease.webhooks.untyped.{EventContentParser, JsonPathEventAccessor, PropertyAccessor}
import grizzled.slf4j.Logging
import org.apache.commons.lang3.StringUtils.isNotBlank
import org.springframework.beans.factory.annotation.Autowired

import java.io.StringWriter
import java.lang.{Boolean => JBool, Double => JDouble}
import java.util
import java.util.{Date, UUID, List => JList}
import javax.script.ScriptContext
import scala.beans.BeanProperty
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer
import scala.jdk.CollectionConverters._
import scala.language.implicitConversions
import scala.util.{Failure, Success, Try}

object EventFilter {
  val FIELD_MAX_LENGTH: Int = 1024
}

@Metadata(virtual = true)
abstract class EventFilter extends BaseConfigurationItem with VariableResolution {
  @Property(asContainment = true)
  var trigger: EventBasedTrigger = _

  def filterId: String = this.getId

  override def folderId: String = Option(this.getId).map(findFolderId).orNull

  def matches(event: Event): Try[Boolean]

  def validate(context: ExtendedValidationContext): Unit

  def formattedValue: String
}

object ScriptBasedEventFilter {
  val RESULT_ATTRIBUTE: String = "result"
}

@Metadata(virtual = true)
abstract class ScriptBasedEventFilter extends EventFilter with Logging {
  @BeanProperty
  @Property(label = "Script expression", description = "Script to filter the event")
  var expr: String = _

  @BeanProperty
  @Autowired
  @transient
  var eventParser: EventContentParser = _

  def scriptExecutor: ScriptExecutor

  override def formattedValue: String = expr

  override def validate(context: ExtendedValidationContext): Unit = {
    if (!isNullOrEmpty(expr) && expr.length > FIELD_MAX_LENGTH) {
      context.error(this, "expr", s"Expression must be $FIELD_MAX_LENGTH characters or less")
    }
  }

  override def matches(event: Event): Try[Boolean] = {
    logger.trace(s"Evaluate filter '$expr' on event $event")
    Option(this.expr) match {
      case Some(expr) if isNotBlank(expr) =>
        Try {
          val context = createContext(event, expr)
          val statementResult = executeScript(context)
          val resultVariable = context.getAttribute(RESULT_ATTRIBUTE)

          (statementResult, resultVariable) match {
            case (null, null) =>
              throw new IllegalArgumentException("Matcher expression did not return anything")
            case (_, resVar: java.lang.Boolean) => resVar.asInstanceOf[Boolean]
            case (stmt: java.lang.Boolean, _) => stmt.asInstanceOf[Boolean]
            case _ =>
              throw new IllegalArgumentException("Matcher expression must return a boolean value")
          }
        }
      case _ => Success(true)
    }
  }

  private def executeScript(scriptContext: XlrScriptContext): AnyRef = {
    val scrubber = SensitiveValueScrubber.disabled

    val logHandler = OutputHandler((l: String) => logger.info(l))
    val executionWriter = new ExecutionOutputWriter(scrubber, new StringWriter(), logHandler)
    JythonSupport.outWriterDecorator.registerWriter(executionWriter)
    scriptContext.setWriter(JythonSupport.outWriterDecorator)

    val errorLogHandler = OutputHandler((l: String) => logger.warn(l))
    val errorExecutionWriter = new ExecutionOutputWriter(scrubber, new StringWriter(), errorLogHandler)
    JythonSupport.errorWriterDecorator.registerWriter(errorExecutionWriter)
    scriptContext.setErrorWriter(JythonSupport.errorWriterDecorator)

    try {
      scriptExecutor.evalScript(scriptContext)
    } finally {
      Try(JythonSupport.outWriterDecorator.close())
      Try(JythonSupport.errorWriterDecorator.close())
      Try(JythonSupport.outWriterDecorator.removeWriter())
      Try(JythonSupport.errorWriterDecorator.removeWriter())
    }
  }


  protected def createContext(event: Event, expr: String): XlrScriptContext
}

object GroovyEventFilter {
  def apply(expr: String): GroovyEventFilter = {
    val filter = new GroovyEventFilter()
    filter.expr = expr
    filter
  }
}

class GroovyEventFilter extends ScriptBasedEventFilter {
  @BeanProperty
  @Autowired
  @transient
  var scriptExecutor: GroovyScriptExecutor = _

  override protected def createContext(event: Event, expr: String): XlrScriptContext = {
    new EventFilterScriptContextBuilder(eventParser, this, event, expr).build()
  }
}

object JythonEventFilter {
  def apply(expr: String): JythonEventFilter = {
    val filter = new JythonEventFilter()
    filter.expr = expr
    filter
  }

  lazy val libraries: Seq[ScriptSource] = Seq(
    ScriptSource.byResource("webhook/JsonSugar.py"),
    ScriptSource.byResource("triggers/JsonGlobals.py")
  )
}

class JythonEventFilter extends ScriptBasedEventFilter {
  @BeanProperty
  @Autowired
  @transient
  var scriptExecutor: JythonScriptExecutor = _

  override protected def createContext(event: Event, expr: String): XlrScriptContext = {
    val builder = new EventFilterScriptContextBuilder(eventParser, this, event, expr)
    libraries.foreach { source =>
      builder.addScript(source)
    }
    builder.build()
  }
}

private class EventFilterScriptContextBuilder(eventParser: EventContentParser,
                                              variableResolution: VariableResolution,
                                              event: Event,
                                              expr: String) extends ScriptContextBuilder {
  private val executionId = UUID.randomUUID().toString
  withLogger().withExecutionId(executionId)

  private val additionalScripts: mutable.Buffer[ScriptSource] = ArrayBuffer()

  def addScript(scriptSource: ScriptSource): ScriptContextBuilder = {
    additionalScripts += scriptSource
    this
  }

  override protected def doBuild(context: XlrScriptContext): Unit = {
    eventParser.parse(event).asScala.foreach {
      case (key, value) => addProperty(context, key, value)
    }
    val variableResolver = variableResolution.variableResolver
    addProperty(context, "folderVariables", variableResolver.folderVars.asScala.view.mapValues(_.getValue).toMap.asJava)
    addProperty(context, "globalVariables", variableResolver.globalVars.asScala.view.mapValues(_.getValue).toMap.asJava)

    additionalScripts.foreach { source =>
      context.addScript(new XlrScript(name=s"<event_filter>[$executionId]", scriptSource = source, checkPermissions = true, wrap = false))
    }
    val frozenExpr = variableResolution.freezeVariables(expr, failIfUnresolved = true)
    context.addScript(XlrScript.byContent(name = s"<event_filter>[$executionId]", content = frozenExpr, wrap= false, checkPermissions = true))
  }

  private def addProperty(context: ScriptContext, name: String, value: Any): Unit = {
    context.setAttribute(name, value, ScriptContext.ENGINE_SCOPE)
  }

}

object ExpressionEventFilterConversions {
  private def jsonParser: JsonProvider = new JacksonJsonProvider()

  implicit def any2String(value: Any): String = if (value == null) {
    ""
  } else {
    VariableHelper.toString(value)
  }

  implicit def any2Date(value: Any): Date = Option(value).map(DateVariableUtils.parseDate).orNull

  implicit def any2Boolean(value: Any): Boolean = value match {
    case null => null.asInstanceOf[Boolean]
    case string: String =>
      if (string.equalsIgnoreCase("true")) {
        true
      } else if (string.equalsIgnoreCase("false")) {
        false
      } else {
        throw new IllegalArgumentException(s"Cannot convert given string '$string' to boolean")
      }
    case _ => value.asInstanceOf[JBool]
  }

  implicit def any2Double(value: Any): Double = value match {
    case null => null.asInstanceOf[Double]
    case string: String => JDouble.parseDouble(string)
    case num: Number => num.doubleValue()
  }

  implicit def anyToList(value: Any): JList[AnyRef] = (value match {
    case null => new util.ArrayList[AnyRef]
    case string: String => jsonParser.parse(string).asInstanceOf[JList[AnyRef]]
    case _ => value.asInstanceOf[JList[AnyRef]]
  }).asScala.map(VariableHelper.toString(_).asInstanceOf[AnyRef]).asJava
}

object ExpressionEventFilter {
  type Predicate = () => Boolean

  def apply(expressions: java.util.List[EventFilterExpressionItem]): ExpressionEventFilter = {
    val filter = new ExpressionEventFilter
    filter.expressions = expressions
    filter
  }

  val ALLOWED_PREFIXES: Seq[String] = Seq("event.", "headers.", "parameters.")

  def eqOps(actual: Any, expected: Any): PartialFunction[EventFilterOperator, Boolean] = {
    {
      case EQUALS => actual == expected
      case NOT_EQUALS => actual != expected
    }
  }

  def boolOps(actual: Boolean, expected: Boolean): PartialFunction[EventFilterOperator, Boolean] = eqOps(actual, expected)

  def dateOps(actual: Date, expected: Date): PartialFunction[EventFilterOperator, Boolean] = {
    eqOps(actual, expected) orElse {
      case AFTER => actual.after(expected)
      case BEFORE => actual.before(expected)
    }
  }

  def numberOps(actual: Double, expected: Double): PartialFunction[EventFilterOperator, Boolean] = {
    eqOps(actual, expected) orElse {
      case GREATER_THAN => actual > expected
      case LESS_THAN => actual < expected
    }
  }

  def textOps(actual: String, expected: String): PartialFunction[EventFilterOperator, Boolean] = {
    eqOps(actual, expected) orElse {
      case INCLUDES => actual.contains(expected)
      case NOT_INCLUDES => !actual.contains(expected)
    }
  }

  def listOps(actual: JList[AnyRef], expected: JList[AnyRef]): PartialFunction[EventFilterOperator, Boolean] = {
    {
      case EQUALS => actual.asScala.toSet == expected.asScala.toSet
      case NOT_EQUALS => actual.asScala.toSet != expected.asScala.toSet
      case CONTAINS_ALL => actual.containsAll(expected)
      case NOT_CONTAINS_ALL => !actual.containsAll(expected)
      case CONTAINS_ANY =>
        val expectedSet = expected.asScala.toSet
        actual.asScala.exists(expectedSet.contains)
      case NOT_CONTAINS_ANY =>
        val expectedSet = expected.asScala.toSet
        !actual.asScala.exists(expectedSet.contains)
    }
  }

  def objOps(actual: AnyRef, expected: Boolean): PartialFunction[EventFilterOperator, Boolean] = {
    {
      case IS_DEFINED => (actual != null) == expected
    }
  }
}

class ExpressionEventFilter extends EventFilter with Logging {

  import ExpressionEventFilter._
  import ExpressionEventFilterConversions._

  @BeanProperty
  @Property(label = "Event filter expressions", description = "List of expressions to be matched to filter the event", asContainment = true)
  var expressions: java.util.List[EventFilterExpressionItem] = _

  @BeanProperty
  @Autowired
  @transient
  var eventParser: EventContentParser = _

  override def validate(context: ExtendedValidationContext): Unit = {
    if (expressions != null) {
      expressions.asScala.foreach(_.validate(context))
    }
  }

  override def matches(event: Event): Try[Boolean] = {
    logger.trace(s"Evaluate filter $expressions on event $event")
    val accessor = new JsonPathEventAccessor(eventParser.parse(event))
    Try(compileExpr(accessor).forall(_ ()))
  }

  def compileExpr(eventAccessor: PropertyAccessor): Seq[Predicate] = {
    expressions.asScala.map(compileItemExpr(eventAccessor, _)).toSeq
  }

  protected def compileItemExpr(eventAccessor: PropertyAccessor, item: EventFilterExpressionItem): Predicate = {
    val actual: Try[Any] = eventAccessor.getProperty(freezeVariables(item.path, failIfUnresolved = true))
    val expected: Any = freezeVariables(item.value, failIfUnresolved = true)
    () => compileValueExpr(item, actual, expected)(item.operator)
  }

  //noinspection ScalaStyle
  private def compileValueExpr(item: EventFilterExpressionItem, actual: Try[Any], expected: Any): PartialFunction[EventFilterOperator, Boolean] = {
    Try(item.valueType match {
      case BOOLEAN => boolOps(actual.get, expected)
      case INTEGER => numberOps(actual.get, expected)
      case STRING => textOps(actual.get, expected)
      case DATE => dateOps(actual.get, expected)
      case LIST_OF_STRING => listOps(actual.get, expected)
      case OBJECT => objOps(any2AnyRef(actual.toOption.orNull), expected)
    }) match {
      case Success(partiallyCompiledExpr) if !partiallyCompiledExpr.isDefinedAt(item.operator) =>
        throw new IllegalArgumentException(s"Operator ${item.operator} is not supported by the ${item.valueType} type")
      case Success(partiallyCompiledExpr) => partiallyCompiledExpr
      case Failure(exception) =>
        throw new IllegalArgumentException(s"Error comparing actual and expected value of ${item.valueType} field '${item.path}'. Reason: ${exception.getMessage}. Expected value '$expected' but the actual value was '${actual.getOrElse("null")}'.")
    }
  }

  override def formattedValue: String = expressions.asScala.map(expr => expr.formattedValue).mkString(" AND ")
}

object EventFilterExpressionItem {
  def apply(valueType: EventFilterValueType, path: String, operator: EventFilterOperator, value: String): EventFilterExpressionItem = {
    val expr = new EventFilterExpressionItem
    expr.path = path
    expr.operator = operator
    expr.valueType = valueType
    expr.value = value
    expr
  }
}

class EventFilterExpressionItem extends BaseConfigurationItem {

  @BeanProperty
  @Property(label = "Path", description = "Path to the json object value")
  var path: String = _

  @BeanProperty
  @Property(label = "Operator", description = "Operator that will test the path value against expression value")
  var operator: EventFilterOperator = _

  @BeanProperty
  @Property(label = "Type", description = "Type of expression value")
  var valueType: EventFilterValueType = _

  @BeanProperty
  @Property(label = "Value", required = false, description = "Expression value")
  var value: String = _

  def validate(context: ExtendedValidationContext): Unit = {
    if (isNullOrEmpty(path)) {
      context.error(this, "path", "Parameter path is missing")
    } else {
      if (path.length > FIELD_MAX_LENGTH) context.error(this, "path", s"Path must be $FIELD_MAX_LENGTH characters or less")
      if (!ALLOWED_PREFIXES.exists(path.startsWith)) {
        context.error(this, "path", s"Path must begin with one of the prefixes: ${ALLOWED_PREFIXES.mkString(", ")}")
      }
    }
    if (operator == null) context.error(this, "operator", "Parameter operator is missing")
    if (valueType == null) context.error(this, "valueType", "Parameter valueType is missing")
    if (!isNullOrEmpty(value) && value.length > FIELD_MAX_LENGTH) context.error(this, "value", s"Value must be $FIELD_MAX_LENGTH characters or less")
  }

  override def toString = s"EventFilterExpressionItem(path=$path, operator=$operator, valueType=$valueType, value=$value)"

  def formattedValue: String = s"$path $operator ${Option(value).getOrElse("''")}"
}

class NoopPropertyAccessor extends PropertyAccessor {
  override def getProperty(path: String): Try[Any] = Success(null)
}
