package com.xebialabs.xlrelease.triggers.repository.persistence

import com.xebialabs.xlrelease.api.v1.filter.TriggerFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.FolderPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate
import org.springframework.util.StringUtils

import scala.jdk.CollectionConverters._

class TriggerQueryBuilder(val folderPersistence: FolderPersistence,
                          val sqlRepositoryAdapter: SqlRepositoryAdapter,
                          val dialect: Dialect,
                          val namedTemplate: NamedParameterJdbcTemplate)
  extends FiltersQueryBuilder[TriggerFilters, Trigger] with FilterQueryBuilderSupport[TriggerFilters, Trigger] with TriggerMapper {

  private lazy val totalQueryTemplate =
    s"""
       |SELECT COUNT(1)
       | FROM ${TRIGGERS.TABLE} t
       | INNER JOIN ${FOLDERS.TABLE} f on t.${TRIGGERS.FOLDER_UID} = f.${FOLDERS.CI_UID}
       | $joinClause
       | $whereClause
    """.stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private lazy val queryTemplate =
    s"""SELECT t.${TRIGGERS.ID},
       | t.${TRIGGERS.FOLDER_UID},
       | t.${TRIGGERS.CI_UID},
       | t.${TRIGGERS.ENABLED},
       | t.${TRIGGERS.CI_TYPE},
       | t.${TRIGGERS.TRIGGER_TITLE},
       | t.${TRIGGERS.DESCRIPTION},
       | t.${TRIGGERS.CONTENT},
       | t.${TRIGGERS.LAST_RUN_DATE},
       | t.${TRIGGERS.LAST_RUN_STATUS},
       | f.${FOLDERS.FOLDER_PATH},
       | f.${FOLDERS.FOLDER_ID}
       | FROM ${TRIGGERS.TABLE} t
       | INNER JOIN ${FOLDERS.TABLE} f on t.${TRIGGERS.FOLDER_UID} = f.${FOLDERS.CI_UID}
       | $joinClause
       | $whereClause
       | $orderClause
       |""".stripMargin.linesIterator.filter(!_.trim.isEmpty).mkString(s"$NL")

  override def from(filter: TriggerFilters): SelfType = {
    if (StringUtils.hasText(filter.getFolderId)) {
      if (filter.getNestedFolders) {
        joinClauses +=
        s"""
             | JOIN ${PATHS.TABLE} p ON p.${PATHS.DESCENDANT_UID} = f.${FOLDERS.CI_UID}
             | JOIN ${FOLDERS.TABLE} af on af.${FOLDERS.CI_UID} = p.${PATHS.ANCESTOR_UID}
             |""".stripMargin
        whereClauses += s"af.${FOLDERS.FOLDER_ID} = :${FOLDERS.FOLDER_ID}"
        orderClauses += s"p.${PATHS.DEPTH} ASC"
      } else {
        whereClauses += s"f.${FOLDERS.FOLDER_ID} = :${FOLDERS.FOLDER_ID}"
      }
      queryParams += FOLDERS.FOLDER_ID -> FolderId(filter.getFolderId).id
    }

    if (StringUtils.hasText(filter.getTemplateId)) {
      joinClauses +=
      s"""
           | JOIN ${TEMPLATE_TRIGGERS.TABLE} tt on tt.${TEMPLATE_TRIGGERS.TRIGGER_UID} = t.${TRIGGERS.CI_UID}
           | JOIN ${RELEASES.TABLE} r ON tt.${TEMPLATE_TRIGGERS.RELEASE_UID} = r.${RELEASES.CI_UID}
           |""".stripMargin
      whereClauses += s"r.${RELEASES.RELEASE_ID} = :templateId"
      queryParams += "templateId" -> Ids.getName(filter.getTemplateId)
    }

    if (!filter.getTriggerType.isEmpty) {
      val triggerTypeQueries = filter.getTriggerType.asScala.zipWithIndex.map {
        case (_, index) => s"t.${TRIGGERS.CI_TYPE} = :triggerType$index" -> index
      }
      whereClauses += triggerTypeQueries.map(_._1).mkString("(", " OR ", ")")
      for (param <- triggerTypeQueries.map(_._2).map(index => s"triggerType$index" -> filter.getTriggerType.get(index))) {
        queryParams += param
      }
    }

    this
  }

  override def build(): PageableQuery[Trigger] = {
    withSortParameters(TRIGGERS.CI_UID -> TRIGGERS.CI_UID)
    buildSortOrderClause(defaultOrderColumn = TRIGGERS.CI_UID)
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[Trigger](namedTemplate, resultsQueryString, queryParams.toMap, triggerMapper[Trigger](sqlRepositoryAdapter, folderPersistence))
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[Trigger](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }


}
