package com.xebialabs.xlrelease.triggers.service

import com.xebialabs.deployit.plugin.api.validation.ValidationMessage
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, TriggerRepository}
import com.xebialabs.xlrelease.service.FolderOperationValidator
import grizzled.slf4j.Logging
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Component

import java.util
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Component
class TriggerFolderOperationValidator(configurationRepository: ConfigurationRepository, triggerRepository: TriggerRepository)
  extends FolderOperationValidator with Logging {

  override def validateMoveOperation(folderId: String, newParentId: String, messages: util.List[ValidationMessage]): Unit = {
    val invalidRefs: collection.Set[BaseConfiguration] = getNonInheritedFolderTriggerReferences(folderId, newParentId)

    if (invalidRefs.nonEmpty) {
      val errorMessage = ValidationMessage.error(
        folderId,
        "configuration references for triggers",
        s"configuration references for triggers not inherited by the destination folder: ${invalidRefs.map(_.getTitle).mkString("\"", "\", \"", "\"")}"
      )

      messages.add(errorMessage)
    }
  }

  override def validateDeleteOperation(folderId: String, messages: util.List[ValidationMessage]): Unit = {
    val all = PageRequest.of(0, Int.MaxValue)
    val triggers = triggerRepository.findByFolderId(folderId, nestedFolders = true, pageable = all).getContent.asScala.toSeq

    val enabledTriggers = triggers.filter(t => t.isEnabled)
    if (enabledTriggers.nonEmpty) {
      val errorMessage = ValidationMessage.error(
        folderId,
        "active triggers",
        s"active triggers, please disable the following triggers before continuing: ${enabledTriggers.map(_.getTitle).mkString("\"", "\", \"", "\"")}"
      )
      messages.add(errorMessage)
    }
  }

  private def getNonInheritedFolderTriggerReferences(folderId: String, newfolderId: String) = {
    val nonInheritedConfIds = configurationRepository.findAllNonInheritedTriggerReferences(folderId)
    val res: mutable.Set[BaseConfiguration] = mutable.Set()

    nonInheritedConfIds.foreach { confId =>
      val c = configurationRepository.read(confId).asInstanceOf[BaseConfiguration]
      if (!(c.getFolderId == null || // is global OK
        newfolderId.startsWith(c.getFolderId) || // is inherited from parent in scope of destination OK
        c.getFolderId.startsWith(folderId))) { // will move along because in same folder or child folder OK
        res.add(c)
      }
    }
    res
  }

}
