package com.xebialabs.xlrelease.triggers.actors

import akka.actor.{Actor, ActorLogging, PoisonPill, Props, ReceiveTimeout}
import akka.cluster.sharding.ShardRegion
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.repository.CiCloneHelper
import com.xebialabs.xlrelease.triggers.actors.TriggerActor._
import com.xebialabs.xlrelease.triggers.service.impl.TriggerExecutionContext
import org.springframework.security.core.Authentication
import org.springframework.security.core.context.SecurityContextHolder

import scala.concurrent.Future
import scala.concurrent.duration._
import scala.util.{Failure, Success, Try}

object TriggerActor {
  def props(clustered: Boolean, triggerActorOperations: TriggerOperations, xlrConfig: XlrConfig): Props =
    Props(new TriggerActor(clustered, triggerActorOperations, xlrConfig)).withDispatcher("xl.dispatchers.release-dispatcher")

  trait TriggerAction {
    def triggerId: String
  }

  abstract class TriggerCommand extends TriggerAction with Serializable {
    val callerContext: Option[Authentication] = Option(SecurityContextHolder.getContext.getAuthentication)
  }

  case class AddTrigger(trigger: Trigger) extends TriggerCommand {
    override def triggerId: String = trigger.getId
  }

  case class RefreshTrigger(trigger: Trigger) extends TriggerCommand {
    override def triggerId: String = trigger.getId
  }

  case class UpdateTrigger(trigger: Trigger) extends TriggerCommand {
    override def triggerId: String = trigger.getId
  }

  case class UpdateVariables(trigger: Trigger) extends TriggerCommand {
    override def triggerId: String = trigger.getId
  }

  case class UpdateInternalState(trigger: Trigger, updateProperties: Seq[String]) extends TriggerCommand {
    override def triggerId: String = trigger.getId
  }

  case class UpdateTriggerStatus(triggerId: String, enabled: Boolean, checkReferencePermissions: Boolean) extends TriggerCommand

  case class AutoDisableTrigger(triggerId: String) extends TriggerCommand

  case class DeleteTrigger(triggerId: String) extends TriggerCommand

  case class ExecuteTrigger(triggerId: String, executionContext: TriggerExecutionContext) extends TriggerCommand

  case class ExecutionStopped(triggerId: String) extends TriggerCommand

  case class ExecuteTriggerResult(result: Either[Trigger, String])
}

class TriggerActor(clustered: Boolean,
                   triggerOperations: TriggerOperations,
                   xlrConfig: XlrConfig)
  extends Actor with ActorLogging {

  private var trigger: Trigger = _

  context.setReceiveTimeout(1.minutes)

  override def receive: Receive = initial

  def initial = withCallerContext(createTrigger orElse manageTrigger orElse executeTrigger orElse passivate)

  def waitingForExecution: Receive = withCallerContext(failCreate orElse manageTrigger orElse executeTrigger orElse passivate)

  def executing: Receive = withCallerContext(failCreate orElse manageTrigger orElse ignoreExecuteTrigger orElse passivate)

  private def createTrigger: Receive = {
    case AddTrigger(trigger) => replyOrFail {
      this.trigger = triggerOperations.addTrigger(trigger)
      context.become(waitingForExecution)
      this.trigger
    }
  }

  private def failCreate: Receive = {
    case AddTrigger(trigger) => replyOrFail {
      throw new IllegalStateException(s"Trigger ${trigger.getId} already exists")
    }
  }

  private def manageTrigger: Receive = {
    case UpdateTrigger(updated) => replyOrFail {
      this.trigger = triggerOperations.updateTrigger(this.trigger, updated, validate = true, emitEvents = true, internal = false, checkReferencePermissions = true, Seq())
      this.trigger
    }

    case UpdateVariables(updated) => replyOrFail {
      this.trigger = triggerOperations.updateTrigger(this.trigger, updated, validate = false, emitEvents = true, internal = true, checkReferencePermissions = true, properties = Seq("variables"))
      this.trigger
    }

    case UpdateInternalState(updated, updateProperties) => replyOrFail {
      this.trigger = triggerOperations.updateTrigger(this.trigger, updated, validate = false, emitEvents = false, internal = true, checkReferencePermissions = true, properties = updateProperties)
      this.trigger
    }

    case UpdateTriggerStatus(_, enabled, checkReferencePermissions) => replyOrFail {
      this.trigger.setEnabled(enabled)
      this.trigger = triggerOperations.updateTrigger(this.trigger, this.trigger, validate = enabled, emitEvents = true, internal = false, checkReferencePermissions = checkReferencePermissions, Seq("enabled"))
      this.trigger
    }

    case AutoDisableTrigger(_) => replyOrFail {
      this.trigger.setEnabled(false)
      this.trigger = triggerOperations.updateTrigger(this.trigger, this.trigger, validate = false, emitEvents = false, internal = false, checkReferencePermissions = true, Seq("enabled"))
      this.trigger
    }

    case RefreshTrigger(trigger) => replyOrFail {
      this.trigger = trigger
      this.trigger
    }

    case DeleteTrigger(_) => replyOrFail {
      triggerOperations.deleteTrigger(this.trigger)
      self ! PoisonPill
    }
  }

  private def executeTrigger: Receive = {
    case ExecuteTrigger(_, executionContext) =>
      // 'event-based triggers' should not discard ExecuteTrigger messages, while 'release triggers' should
      if (!trigger.getProperty[Boolean](Trigger.ALLOW_PARALLEL_EXECUTION)) {
        context.become(executing)
      }
      val clonedTrigger = CiCloneHelper.cloneCi(this.trigger)
      clonedTrigger.setCiUid(this.trigger.getCiUid)
      executeAsync(clonedTrigger, executionContext)
    case ExecutionStopped(_) => ()
  }

  private def executeAsync(clonedTrigger: Trigger, executionContext: TriggerExecutionContext): Unit = {
    val originalSender = sender()
    val deadLetters = context.system.deadLetters
    val triggerId = clonedTrigger.getId
    implicit val ec = xlrConfig.executors.releaseTrigger.executionContext
    Future {
      val res: Trigger = if (executionContext.shouldExecute(clonedTrigger)) {
        triggerOperations.execute(clonedTrigger, executionContext)
      } else {
        log.warning(s"Trigger ${clonedTrigger.getId} was disabled in the meantime, will skip execution.")
        // returning a trigger here results in event message being acknowledged for event-based triggers
        clonedTrigger
      }
      res
    } onComplete { result =>
      if (!result.isSuccess) {
        val ex = result.failed.get
        log.error(ex, s"Trigger `${clonedTrigger.getId}` failed with `${result.failed.get.getMessage}`");
      }
      self ! ExecutionStopped(triggerId)
      if (originalSender != deadLetters) {
        val response = ExecuteTriggerResult(result match {
          case Success(trigger) => scala.util.Left(trigger)
          case Failure(exception) => 
            log.error(exception, "trigger {} execution failed", clonedTrigger.getId)
            scala.util.Right(s"${exception.getClass.getCanonicalName}: ${exception.getMessage}")
        })
        originalSender ! response
      }
    }
  }

  private def ignoreExecuteTrigger: Receive = {
    case ExecuteTrigger(_, _) => ()
    case ExecutionStopped(_) => context.become(waitingForExecution)
  }

  private def passivate: Receive = {
    case ReceiveTimeout if clustered => context.parent ! ShardRegion.Passivate(PoisonPill)
    case ReceiveTimeout if !clustered => context.stop(self)
    case PoisonPill => context.stop(self)
  }

  private def replyOrFail[T](call: => T): Unit = sender() ! (Try(call) match {
    case Success(t) if t != null => t
    case Success(_) => akka.actor.Status.Failure(new NullPointerException("Method returned null and this cannot be processed"))
    case Failure(ex) =>
      log.error(ex, "Failed to process trigger message")
      akka.actor.Status.Failure(ex)
  })

  private def withCallerContext(delegate: Receive): Receive = {
    case cmd: AddTrigger =>
      withAuthentication(delegate, cmd)
    case cmd: TriggerCommand =>
      if (this.trigger == null) {
        this.trigger = triggerOperations.find(cmd.triggerId)
      }
      withAuthentication(delegate, cmd)
    case msg@_ =>
      delegate(msg)
  }

  private def withAuthentication(delegate: Receive, cmd: TriggerCommand) = {
    val maybeAuthentication = cmd.callerContext
    log.debug(s"Setting caller as ${maybeAuthentication.map(_.getName)} for command $cmd")
    SecurityContextHolder.getContext.setAuthentication(maybeAuthentication.orNull)
    try {
      delegate(cmd)
    } finally {
      SecurityContextHolder.clearContext()
    }
  }
}
