package com.xebialabs.xlrelease.triggers.events.handler

import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.{ReleaseTrigger, Trigger}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository.TriggerRepository
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.TriggerConfigurationReferencePersistence
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.triggers.events.handler.TriggerEventHandler.{DELETE_OPERATION, MOVE_OPERATION}
import com.xebialabs.xlrelease.triggers.service.TriggerService
import com.xebialabs.xlrelease.triggers.service.impl.TriggerLifecycle
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds
import grizzled.slf4j.Logging
import org.springframework.data.domain.PageRequest

import scala.collection.mutable
import scala.jdk.CollectionConverters._

object TriggerEventHandler {
  val MOVE_OPERATION = "move"
  val DELETE_OPERATION = "delete"
}

@EventListener
class TriggerEventHandler(triggerRepository: TriggerRepository,
                          triggerConfigurationReferencePersistence: TriggerConfigurationReferencePersistence,
                          ciIdService: CiIdService,
                          triggerService: TriggerService,
                          val triggerLifecycle: TriggerLifecycle[Trigger]
                         ) extends Logging {

  @Subscribe
  def onTemplateVariableSync(event: TemplateVariablesChangedEvent): Unit = {
    val templateId = event.templateId
    val releaseTriggers = triggerRepository.findByTemplateId(templateId, PageRequest.of(0, Int.MaxValue)).getContent.asScala
      .filter(_.isInstanceOf[ReleaseTrigger])
      .map(_.asInstanceOf[ReleaseTrigger])
    for (trigger <- releaseTriggers) {
      val templateVarsMap = mutable.Map(event.templateVariables.asScala.map(v => v.getKey -> cloneCi(v)).toSeq: _*)
      // remove variables from trigger that don't belong to template
      val triggerVariablesNeededForTemplate = trigger.getVariables.asScala.filter(trigVar => templateVarsMap.contains(trigVar.getKey))
      // update values of existing variables and remove updated ones from templateVarsMap
      val updatedTriggerVars = triggerVariablesNeededForTemplate.flatMap { triggerVar =>
        templateVarsMap.remove(triggerVar.getKey) map { templateVar =>
          // Keep trigger's variable value if it's not empty, otherwise variable's default value will be set
          if (!triggerVar.isValueEmpty) {
            templateVar.setUntypedValue(triggerVar.getValue)
          }
          templateVar
        }
      }
      val newVariables = templateVarsMap.values
      val newTriggerVars = (updatedTriggerVars ++ newVariables).asJava
      fixUpVariableIds(trigger.getId, newTriggerVars, ciIdService)
      trigger.setVariables(newTriggerVars)
      triggerService.updateVariables(trigger)
    }
  }

  @Subscribe
  def onConfigurationUpdated(event: ConfigurationUpdatedEvent): Unit = {
    val triggers = triggerConfigurationReferencePersistence.getReferencingEntities(event.updated.getId)
    triggers.foreach { triggerRow =>
      val trigger = triggerService.findById(triggerRow.id)
      triggerService.refreshTrigger(trigger)
    }
  }

  @Subscribe
  def onFolderMoving(action: FolderMovingAction): Unit = {
    checkFolderOperation(action, MOVE_OPERATION)
  }

  @Subscribe
  def onFolderDeleting(action: FolderDeletingAction): Unit = {
    checkFolderOperation(action, DELETE_OPERATION)
  }

  @Subscribe
  def onTemplateMoving(action: TemplateMovingAction): Unit = {
    val numberOfTemplateTriggers = triggerRepository.numberOfTemplateTriggers(action.templateId)
    if (numberOfTemplateTriggers > 0) {
      logger.warn(s"Tried to move template ${action.templateId} with triggers")
      throw new ItemInUseException(s"You cannot move the template because it is being used by one or more triggers")
    }
  }

  @Subscribe
  def onTemplateDeleting(action: TemplateDeletingAction): Unit = {
    val numberOfTemplateTriggers = triggerRepository.numberOfTemplateTriggers(action.templateId)
    if (numberOfTemplateTriggers > 0) {
      triggerRepository
        .findByTemplateId(action.templateId, PageRequest.of(0, Int.MaxValue))
        .forEach(t => triggerService.deleteTrigger(t.getId))
    }
  }

  private def checkFolderOperation(action: FolderAction, operation: String): Unit = {
    val all = PageRequest.of(0, Int.MaxValue)
    val triggers = triggerRepository.findByFolderId(action.folderId, nestedFolders = true, pageable = all).getContent.asScala.toSeq
    operation match {
      case MOVE_OPERATION =>
        checkTriggers(triggers, s"You cannot $MOVE_OPERATION this folder. The folder or its subfolders contain triggers")
      case DELETE_OPERATION =>
        checkTriggers(triggers.filter(t => t.isEnabled), s"You cannot $DELETE_OPERATION this folder. " +
          s"The folder or its subfolders contain active triggers, please disable the following triggers before continuing")
        triggers.foreach(t => triggerService.deleteTrigger(t.getId))
    }
  }

  private def checkTriggers(triggers: Seq[Trigger], msg: String): Unit = {
    if (triggers.nonEmpty) {
      val titles = triggers.map(t => t.getTitle).mkString("\"", "\", \"", "\"")
      throw new ItemInUseException(
        s"$msg: $titles"
      )
    }
  }
}
