package com.xebialabs.xlrelease.triggers.scheduled.quartz

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.configuration.TriggerDataPurgeSettings
import com.xebialabs.xlrelease.configuration.TriggerDataPurgeSettings.TRIGGER_DATA_PURGE_SETTINGS_ID
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.events.ConfigurationUpdatedEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration.BEAN_QUARTZ_JOB_SCHEDULER
import com.xebialabs.xlrelease.quartz.events.SchedulerStartedEvent
import com.xebialabs.xlrelease.service.ConfigurationService
import com.xebialabs.xlrelease.triggers.scheduled.quartz.TriggerDataPurgeJob.DATA_PURGE_AGE_KEY
import com.xebialabs.xlrelease.triggers.scheduled.quartz.TriggerDataPurgeService.{TRIGGER_DATA_PURGE_SERVICE, triggerPurgeDataSettingsType}
import grizzled.slf4j.Logging
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{JobDataMap, JobDetail, Scheduler, SimpleTrigger}
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.event
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

object TriggerDataPurgeService {
  val TRIGGER_DATA_PURGE_SERVICE = "Triggers data purge"
  val triggerPurgeDataSettingsType = Type.valueOf(classOf[TriggerDataPurgeSettings])
}

@Service
@EventListener
@IsTransactional
class TriggerDataPurgeService(@Qualifier(BEAN_QUARTZ_JOB_SCHEDULER) val scheduler: Scheduler,
                               val configuration: ConfigurationService
                             ) extends Logging {

  private val serviceName = TRIGGER_DATA_PURGE_SERVICE
  private val jobClazz = classOf[TriggerDataPurgeJob]
  private val triggerDataPurgeJobInterval = XlrConfig.getInstance.durations.triggerDataPurgeJobInterval

  private val jobName: String = s"$serviceName JOB".toUpperCase.split(' ').mkString("_")
  private val groupName: String = serviceName.toUpperCase.split(' ').mkString("_")

  lazy val job: JobDetail = newJob(jobClazz)
    .withDescription(s"$serviceName job")
    .withIdentity(jobName, groupName)
    .storeDurably(true)
    .setJobData(
      new JobDataMap(
        Map[String, Int](DATA_PURGE_AGE_KEY -> getTriggerPurgeDataSettings.getTriggersDataPurgeAgeThreshold).asJava
      )
    )
    .build()

  lazy val trigger: SimpleTrigger = newTrigger()
    .withIdentity(jobName, groupName)
    .withDescription(s"Trigger $serviceName jobs")
    .withSchedule(
      simpleSchedule().withIntervalInMilliseconds(triggerDataPurgeJobInterval.toMillis).repeatForever()
        .withMisfireHandlingInstructionNowWithRemainingCount()
    )
    .startNow()
    .build()

  def resume(): Unit = {
    scheduler.resumeJob(job.getKey)
    logger.debug(s"Started $serviceName service")
  }

  def pause(): Unit = {
    scheduler.pauseJob(job.getKey)
    logger.debug(s"Stopped $serviceName service")
  }

  private def schedule(): Unit = {
    logger.debug(s"Scheduled $serviceName service")
    job.getJobDataMap.put(DATA_PURGE_AGE_KEY, getTriggerPurgeDataSettings.getTriggersDataPurgeAgeThreshold)
    scheduler.addJob(job, true)

    if (scheduler.checkExists(trigger.getKey)) {
      scheduler.rescheduleJob(trigger.getKey, trigger)
    } else {
      scheduler.scheduleJob(job, Set(trigger).asJava, true)
    }
  }

  private def unschedule(): Unit = {
    logger.debug(s"Unscheduled $serviceName service")
    if (scheduler.checkExists(trigger.getKey)) {
      scheduler.unscheduleJob(trigger.getKey)
    }
  }

  private def getTriggerPurgeDataSettings: TriggerDataPurgeSettings = {
    configuration.read(TRIGGER_DATA_PURGE_SETTINGS_ID).asInstanceOf[TriggerDataPurgeSettings]
  }

  @event.EventListener
  def onStartup(event: SchedulerStartedEvent): Unit = {
    scheduleOrUnschedule()
  }

  @AsyncSubscribe
  def onConfigurationUpdate(event: ConfigurationUpdatedEvent): Unit = {
    if (event.updated.getType.instanceOf(triggerPurgeDataSettingsType)) {
      scheduleOrUnschedule()
    }
  }

  private def scheduleOrUnschedule(): Unit = {
    try {
      if (getTriggerPurgeDataSettings.isEnabled) {
        schedule()
      } else {
        unschedule()
      }
    } catch {
      case e: Exception => logger.error(s"Error while starting $serviceName service", e)
    }
  }

}
