package com.xebialabs.xlrelease.triggers.event_based

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.webhooks.domain.HttpRequestEvent
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.triggers.events.{TriggerFilterFailedEvent, TriggerFilterPassedEvent}
import com.xebialabs.xlrelease.triggers.service.TriggerService
import com.xebialabs.xlrelease.webhooks.consumers.BaseConsumerHandler
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.ApplicationContext
import org.springframework.stereotype.Component

import scala.util.{Failure, Success}

@Component
class EventBasedTriggerConsumerHandler @Autowired()(triggerService: TriggerService,
                                                    applicationContext: ApplicationContext,
                                                    eventBus: XLReleaseEventBus)
  extends BaseConsumerHandler[EventBasedTriggerConsumer]
    with Logging
    with EventFiltering {

  override def consumerConfigType: Type = Type.valueOf(classOf[EventBasedTriggerConsumer])

  override def filter(config: EventBasedTriggerConsumer, event: Event): Boolean = {
    event match {
      case httpEvent: HttpRequestEvent if config.enabled && config._eventType == Type.valueOf(classOf[HttpRequestEvent]) =>
        val trigger = triggerService.findById(config.targetId).asInstanceOf[EventBasedTrigger]
        try {
          val result = doFilter(trigger, httpEvent, applicationContext)
          eventBus.publish(TriggerFilterPassedEvent(trigger, event.getId))
          result
        } catch {
          case e: Throwable =>
            eventBus.publish(TriggerFilterFailedEvent(trigger, e.getMessage, event.getId))
            throw e
        }
      case _ => true
    }
  }

  override def consumeEvent(triggerConfig: EventBasedTriggerConsumer, event: Event): Boolean = {
    triggerService.executeSync(triggerConfig.targetId, EventBasedTriggerExecutionContext(event))
    true
  }

}

trait EventFiltering {
  self: Logging =>

  protected def doFilter(trigger: EventBasedTrigger, httpEvent: HttpRequestEvent, applicationContext: ApplicationContext): Boolean = {
    Option(trigger.eventFilter).fold {
      logger.info(s"Executing trigger '${trigger.getTitle}' on new event")
      true
    } { eventFilter =>
      applicationContext.getAutowireCapableBeanFactory.autowireBean(eventFilter)
      eventFilter.matches(httpEvent) match {
        case Success(false) =>
          logger.debug(s"Filter not satisfied on trigger '${trigger.getTitle}' [${trigger.getId}]")
          false
        case Success(true) =>
          logger.debug(s"Filter satisfied on trigger '${trigger.getTitle}' [${trigger.getId}]")
          logger.info(s"Executing trigger '${trigger.getTitle}' on new matched event")
          true
        case Failure(exception) =>
          logger.warn(s"Error evaluating filter of trigger '${trigger.getTitle}' [${trigger.getId}]")
          throw exception
      }
    }
  }
}